<?php
// ═══════════════════════════════════════════════════════════════
//  SIGEDEC — finance/salaries.php
//  Gestão de salários dos líderes/pastores a tempo inteiro.
//  Apenas o finance_user e church_admin podem aceder e editar.
// ═══════════════════════════════════════════════════════════════
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/layout.php';

$user = currentUser();
if (!$user || !in_array($user['role'], ['finance_user','admin'])) {
    header('Location: /index.php'); exit;
}

$churchId    = $user['church_id'];
$church      = getChurchInfo($churchId);
$leadersData = getLeaders($churchId);
$lang        = getLang();

// ── Guardar salário ────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_salary') {
    $lid    = trim($_POST['leader_id'] ?? '');
    $salary = $_POST['salary'] === '' ? null : (float)str_replace(',', '.', $_POST['salary'] ?? '');
    $notes  = trim($_POST['salary_notes'] ?? '');

    foreach ($leadersData['leaders'] as &$l) {
        if ($l['id'] === $lid) {
            $l['salary']       = $salary;
            $l['salary_notes'] = $notes;
            $l['salary_updated_by'] = $user['name'];
            $l['salary_updated_at'] = date('Y-m-d H:i:s');
            break;
        }
    }
    saveLeaders($churchId, $leadersData);
    flashMessage('success', $lang === 'en' ? 'Salary updated.' : 'Salário actualizado.');
    header('Location: salaries.php'); exit;
}

// ── Dados para a página ────────────────────────────────────────
$leaders = $leadersData['leaders'];
$totalSalaries = array_sum(array_filter(array_column($leaders, 'salary')));

renderHead(($lang==='en' ? 'Leader Salaries' : 'Salários dos Líderes') . ' — ' . $church['name'], $church);
?>
<body>
<?php renderSidebar($user['role'], $church); ?>
<div class="main-content">
    <div class="topbar">
        <div class="topbar-left">
            <div class="topbar-title">💰 <?= $lang==='en' ? 'Leader Salaries' : 'Salários dos Líderes' ?></div>
            <div class="topbar-sub"><?= sanitize($church['name']) ?></div>
        </div>
    </div>
    <div class="page-content">
        <?php renderSuggestionBanner(); ?>
        <?php renderFlash(); ?>

        <!-- Totais -->
        <?php if (!empty($leaders)): ?>
        <div class="stats-grid" style="grid-template-columns:repeat(3,1fr);margin-bottom:1.5rem">
            <div class="stat-card">
                <div class="stat-icon">👥</div>
                <div class="stat-info">
                    <div class="stat-value"><?= count($leaders) ?></div>
                    <div class="stat-label"><?= $lang==='en' ? 'Total Leaders' : 'Total Líderes' ?></div>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">⏱️</div>
                <div class="stat-info">
                    <div class="stat-value"><?= count(array_filter($leaders, fn($l) => !empty($l['full_time']))) ?></div>
                    <div class="stat-label"><?= $lang==='en' ? 'Full-Time' : 'Tempo Inteiro' ?></div>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">💰</div>
                <div class="stat-info">
                    <div class="stat-value"><?= number_format($totalSalaries, 2, ',', '.') ?></div>
                    <div class="stat-label"><?= $lang==='en' ? 'Total Salaries (MT/month)' : 'Total Salários (MT/mês)' ?></div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <div class="card">
            <div class="card-header">
                <h3 class="card-title">👑 <?= $lang==='en' ? 'Salary per Leader' : 'Salário por Líder' ?></h3>
                <span class="badge badge-gold"><?= count($leaders) ?></span>
            </div>

            <?php if (empty($leaders)): ?>
            <div class="empty-state">
                <div class="empty-icon">👑</div>
                <h3><?= $lang==='en' ? 'No leaders registered' : 'Sem líderes registados' ?></h3>
                <p><?= $lang==='en' ? 'Add leaders in the Leaders section first.' : 'Adicione líderes na secção de Líderes primeiro.' ?></p>
            </div>
            <?php else: ?>
            <div class="table-wrap">
                <table>
                    <thead>
                        <tr>
                            <th><?= $lang==='en' ? 'Leader' : 'Líder' ?></th>
                            <th><?= $lang==='en' ? 'Role' : 'Cargo' ?></th>
                            <th><?= $lang==='en' ? 'Dedication' : 'Dedicação' ?></th>
                            <th><?= $lang==='en' ? 'Monthly Salary (MT)' : 'Salário Mensal (MT)' ?></th>
                            <th><?= $lang==='en' ? 'Notes' : 'Observações' ?></th>
                            <th><?= $lang==='en' ? 'Last Updated' : 'Última Act.' ?></th>
                            <th><?= $lang==='en' ? 'Action' : 'Acção' ?></th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($leaders as $l):
                        $mname = getMemberName($churchId, $l['member_id']);
                        $initial = strtoupper(substr($mname, 0, 1));
                        $isFullTime = !empty($l['full_time']);
                    ?>
                    <tr>
                        <td>
                            <div class="member-info">
                                <div class="member-avatar" style="background:linear-gradient(135deg,#c9a84c,#f5d78e)"><?= $initial ?></div>
                                <div style="font-weight:600"><?= sanitize($mname) ?></div>
                            </div>
                        </td>
                        <td><span class="badge badge-gold">👑 <?= sanitize($l['role'] ?: 'Líder') ?></span></td>
                        <td>
                            <?php if ($isFullTime): ?>
                            <span class="badge" style="background:#dcfce7;color:#16a34a">⏱️ <?= $lang==='en' ? 'Full-Time' : 'Tempo Inteiro' ?></span>
                            <?php else: ?>
                            <span class="badge" style="background:#f1f5f9;color:#64748b">⏳ <?= $lang==='en' ? 'Part-Time' : 'Parcial' ?></span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if (isset($l['salary']) && $l['salary'] !== null): ?>
                            <strong style="color:var(--gold)"><?= number_format((float)$l['salary'], 2, ',', '.') ?></strong>
                            <?php else: ?>
                            <span class="text-muted" style="font-size:.85rem"><?= $lang==='en' ? 'Volunteer (no salary)' : 'Voluntário (sem salário)' ?></span>
                            <?php endif; ?>
                        </td>
                        <td class="text-sm text-muted"><?= sanitize($l['salary_notes'] ?? '—') ?></td>
                        <td class="text-xs text-muted">
                            <?= !empty($l['salary_updated_at'])
                                ? date('d/m/Y', strtotime($l['salary_updated_at'])) . '<br><span style="font-size:.7rem">' . sanitize($l['salary_updated_by'] ?? '') . '</span>'
                                : '—' ?>
                        </td>
                        <td>
                            <button class="btn btn-outline btn-sm"
                                onclick="openSalaryModal('<?= $l['id'] ?>','<?= addslashes(sanitize($mname)) ?>','<?= $l['salary'] ?? '' ?>','<?= addslashes(sanitize($l['salary_notes'] ?? '')) ?>')">
                                ✏️ <?= $lang==='en' ? 'Edit' : 'Editar' ?>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>

        <p class="text-sm text-muted" style="margin-top:1rem;text-align:center">
            <?= $lang==='en'
                ? 'Salary information is confidential. Only finance team members have access to this page.'
                : 'A informação salarial é confidencial. Apenas membros da equipa financeira têm acesso a esta página.' ?>
        </p>
    </div>
</div>

<!-- Modal: Editar Salário -->
<div class="modal-overlay" id="modal-salary">
    <div class="modal" style="max-width:440px">
        <div class="modal-header">
            <h3 class="modal-title">💰 <span id="salary-modal-name"></span></h3>
            <button class="modal-close">✕</button>
        </div>
        <form method="POST">
            <input type="hidden" name="action" value="save_salary">
            <input type="hidden" name="leader_id" id="salary-leader-id">
            <div class="modal-body">
                <div class="form-group">
                    <label><?= $lang==='en' ? 'Monthly Salary (MT) — leave empty for volunteer' : 'Salário Mensal (MT) — deixe vazio para voluntário' ?></label>
                    <input type="number" name="salary" id="salary-value" class="form-control"
                           step="0.01" min="0" placeholder="<?= $lang==='en' ? '0.00 — volunteer if empty' : '0.00 — voluntário se vazio' ?>">
                </div>
                <div class="form-group">
                    <label><?= $lang==='en' ? 'Notes (optional)' : 'Observações (opcional)' ?></label>
                    <input type="text" name="salary_notes" id="salary-notes" class="form-control"
                           placeholder="<?= $lang==='en' ? 'e.g. Includes housing allowance' : 'ex: Inclui subsídio de habitação' ?>">
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-outline modal-close"><?= $lang==='en' ? 'Cancel' : 'Cancelar' ?></button>
                <button type="submit" class="btn btn-primary">💾 <?= $lang==='en' ? 'Save Salary' : 'Guardar Salário' ?></button>
            </div>
        </form>
    </div>
</div>

<?php renderJS(); ?>
<script>
function openSalaryModal(id, name, salary, notes) {
    document.getElementById('salary-leader-id').value = id;
    document.getElementById('salary-modal-name').textContent = name;
    document.getElementById('salary-value').value = salary;
    document.getElementById('salary-notes').value = notes;
    openModal('modal-salary');
}
</script>
</body>
</html>
