<?php
require_once __DIR__ . '/includes/config.php';

if (isLoggedIn()) {
    $user = currentUser();
    switch ($user['role']) {
        case 'super_admin':  header('Location: admin/');   break;
        case 'church_admin': header('Location: church/');  break;
        case 'local_user':   header('Location: local/');   break;
        case 'finance_user': header('Location: finance/'); break;
        case 'member':       header('Location: member/');  break;
        default:             header('Location: index.php?bye=1'); break;
    }
    exit;
}

$error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = strtolower(trim($_POST['email'] ?? ''));
    $pass  = $_POST['password'] ?? '';

    if (!$email || !$pass) {
        $error = __('err_required');
    } else {
        // Super-admin
        $sys = readJSON(SYSTEM_FILE);
        $sa  = $sys['super_admin'] ?? [];
        if (strtolower($sa['email'] ?? '') === $email && password_verify($pass, $sa['password'] ?? '')) {
            $_SESSION['sigedec_user'] = ['role'=>'super_admin','email'=>$email,'name'=>$sa['name'],'church_id'=>null];
            header('Location: admin/'); exit;
        }
        // Igrejas
        foreach (getAllChurches() as $cid => $c) {
            $admin = $c['admin'] ?? [];
            if (strtolower($admin['email'] ?? '') === $email && password_verify($pass, $admin['password'] ?? '')) {
                $_SESSION['sigedec_user'] = ['role'=>'church_admin','email'=>$email,'name'=>$admin['name'],'church_id'=>$cid,'church_name'=>$c['name']];
                header('Location: church/'); exit;
            }
            foreach ((getChurchUsers($cid)['users'] ?? []) as $u) {
                if (strtolower($u['email'] ?? '') === $email && password_verify($pass, $u['password'] ?? '')) {
                    $_SESSION['sigedec_user'] = ['role'=>$u['role'],'email'=>$email,'name'=>$u['name'],'church_id'=>$cid,'church_name'=>$c['name'],'subdivision_id'=>$u['subdivision_id']??null];
                    header('Location: ' . ($u['role']==='finance_user' ? 'finance/' : 'local/')); exit;
                }
            }
        }
        // Member portal login
        $memberMatch = findMemberByPortalEmail($email);
        if ($memberMatch) {
            $m = $memberMatch['member'];
            if (!empty($m['portal_password']) && password_verify($pass, $m['portal_password'])) {
                $_SESSION['sigedec_user'] = [
                    'role'        => 'member',
                    'email'       => $m['portal_email'],
                    'name'        => $m['name'],
                    'church_id'   => $memberMatch['church_id'],
                    'church_name' => $memberMatch['church']['name'] ?? '',
                    'member_id'   => $m['id'],
                ];
                header('Location: member/');
                exit;
            }
        }
        $error = __('login_error');
    }
}

$year  = date('Y');
$month = date('m');
$day   = date('d');
$monthNames = ['pt'=>['Janeiro','Fevereiro','Março','Abril','Maio','Junho','Julho','Agosto','Setembro','Outubro','Novembro','Dezembro'],
               'en'=>['January','February','March','April','May','June','July','August','September','October','November','December']];
$monthName = $monthNames[$lang][(int)$month - 1];
?>
<!DOCTYPE html>
<html lang="<?= $lang ?>">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>SIGEDEC — <?= $lang==='en' ? 'African Church Management System' : 'Sistema de Gestão Eclesiástica para África' ?></title>
<meta name="description" content="<?= $lang==='en' ? 'SIGEDEC is the leading church management system built for Africa. Manage members, finances, ministries and more across all denominations securely.' : 'O SIGEDEC é o principal sistema de gestão eclesiástica desenvolvido para África. Gira membros, finanças, ministérios e muito mais em total segurança.' ?>">
<meta name="keywords" content="church management africa, gestão igreja africa, sistema gestão eclesiástica, church software mozambique, angola, kenya, nigeria, ghana, members church system">
<meta name="robots" content="index, follow">
<meta name="author" content="SIGEDEC — sigedec.asaf.fyi">
<meta property="og:title" content="SIGEDEC — African Church Management System">
<meta property="og:description" content="<?= $lang==='en' ? 'Manage your African denomination digitally. Members, finance, ministries and reports — all in one place.' : 'Gira a tua denominação africana de forma digital. Membros, finanças, ministérios e relatórios — tudo num só lugar.' ?>">
<meta property="og:type" content="website">
<meta property="og:url" content="<?= SITE_URL ?>">
<meta property="og:image" content="<?= SITE_URL ?>/assets/img/og-image.png">
<meta name="twitter:card" content="summary_large_image">
<link rel="canonical" href="<?= SITE_URL ?>">
<link rel="sitemap" type="application/xml" href="<?= SITE_URL ?>/sitemap.xml">
<link rel="stylesheet" href="assets/css/style.css">
<link rel="stylesheet" href="assets/css/landing.css">
<link rel="icon" type="image/svg+xml" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>✝</text></svg>">
<!-- Structured Data -->
<script type="application/ld+json">
{
  "@context":"https://schema.org",
  "@type":"SoftwareApplication",
  "name":"SIGEDEC",
  "description":"Church management system for Africa",
  "url":"<?= SITE_URL ?>",
  "applicationCategory":"BusinessApplication",
  "operatingSystem":"Web",
  "offers":{"@type":"Offer","price":"0","priceCurrency":"USD","description":"6 months free trial"},
  "author":{"@type":"Organization","name":"SIGEDEC","email":"<?= CONTACT_EMAIL ?>"},
  "availableLanguage":["Portuguese","English"],
  "areaServed":"Africa"
}
</script>
</head>
<body class="landing-body">

<!-- ═══ TOPO / HEADER ════════════════════════════════════════════════ -->
<header class="landing-header">
    <div class="land-container land-header-inner">
        <div class="land-logo">
            <span class="land-cross">✝</span>
            <span class="land-logo-text">SIGEDEC</span>
        </div>
        <nav class="land-nav">
            <a href="#sobre"><?= $lang==='en' ? 'About' : 'Sobre' ?></a>
            <a href="#funcionalidades"><?= $lang==='en' ? 'Features' : 'Funcionalidades' ?></a>
            <a href="#precos"><?= $lang==='en' ? 'Pricing' : 'Preços' ?></a>
        <a href="#standalone"><?= $lang==='en' ? 'Standalone' : 'Standalone' ?></a>
            <a href="#tutorial">Tutorial</a>
            <a href="#faq">FAQ</a>
            <a href="#contacto"><?= $lang==='en' ? 'Contact' : 'Contacto' ?></a>
            <a href="public/"><?= $lang==='en' ? 'Directory' : 'Directório' ?></a>
        </nav>
        <div class="land-header-actions">
            <a href="?lang=<?= __('lang_code') ?>" class="btn-lang">🌐 <?= __('lang_switch') ?></a>
            <a href="register.php" class="btn-reg"><?= __('register') ?></a>
        </div>
        <button class="land-menu-btn" id="landMenuBtn">☰</button>
    </div>
    <!-- Mobile nav -->
    <div class="land-mobile-nav" id="landMobileNav">
        <a href="#sobre"><?= $lang==='en' ? 'About' : 'Sobre' ?></a>
        <a href="#funcionalidades"><?= $lang==='en' ? 'Features' : 'Funcionalidades' ?></a>
        <a href="#precos"><?= $lang==='en' ? 'Pricing' : 'Preços' ?></a>
        <a href="#standalone"><?= $lang==='en' ? 'Standalone' : 'Standalone' ?></a>
        <a href="#tutorial">Tutorial</a>
        <a href="#faq">FAQ</a>
        <a href="register.php"><?= __('register') ?></a>
        <a href="?lang=<?= __('lang_code') ?>">🌐 <?= __('lang_switch') ?></a>
    </div>
</header>

<!-- ═══ HERO + LOGIN ════════════════════════════════════════════════ -->
<section class="land-hero" id="login">
    <div class="land-container land-hero-inner">
        <!-- Coluna Hero -->
        <div class="land-hero-text">
            <div class="land-hero-badge">
                <?= $lang==='en' ? '🌍 Built for Africa' : '🌍 Desenvolvido para África' ?>
            </div>
            <h1 class="land-h1">
                <?= $lang==='en'
                    ? 'Your Denomination, <span>Organised.</span>'
                    : 'A sua Denominação, <span>Organizada.</span>' ?>
            </h1>
            <p class="land-hero-sub">
                <?= $lang==='en'
                    ? 'SIGEDEC is the complete digital management system designed specifically for African churches and denominations. Manage members, finances, ministries, reports and much more — from any device, anywhere in Africa.'
                    : 'O SIGEDEC é o sistema digital completo desenhado especificamente para igrejas e denominações africanas. Gira membros, finanças, ministérios, relatórios e muito mais — de qualquer dispositivo, em qualquer lugar de África.' ?>
            </p>
            <div class="land-hero-stats">
                <div class="land-hero-stat"><strong>6</strong><span><?= $lang==='en' ? 'months free' : 'meses grátis' ?></span></div>
                <div class="land-hero-stat"><strong>100%</strong><span><?= $lang==='en' ? 'data privacy' : 'privacidade de dados' ?></span></div>
                <div class="land-hero-stat"><strong>2</strong><span><?= $lang==='en' ? 'languages' : 'idiomas' ?></span></div>
            </div>
            <a href="register.php" class="btn btn-primary btn-lg"><?= __('reg_btn') ?></a>
        </div>
        <!-- Login Card -->
        <div class="land-login-card">
            <div class="land-login-header">
                <div class="land-login-cross">✝</div>
                <h2><?= __('login_welcome') ?></h2>
                <p><?= __('login_subtitle') ?></p>
            </div>
            <?php if ($error): ?>
            <div class="alert alert-danger" style="margin:0 1.5rem 1rem">❌ <?= htmlspecialchars($error) ?></div>
            <?php endif; ?>
            <?php if (isset($_GET['expired'])): ?>
            <div class="alert alert-warning" style="margin:0 1.5rem 1rem">⏰ <?= __('session_expired') ?></div>
            <?php endif; ?>
            <?php if (isset($_GET['bye'])): ?>
            <div class="alert alert-info" style="margin:0 1.5rem 1rem">👋 <?= __('logged_out') ?></div>
            <?php endif; ?>
            <form method="POST" class="land-login-form">
                <div class="form-group">
                    <label><?= __('email') ?></label>
                    <input type="email" name="email" class="form-control" placeholder="denominacao@email.com"
                        value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required autofocus>
                </div>
                <div class="form-group">
                    <label><?= __('password') ?></label>
                    <div class="pass-wrap">
                        <input type="password" name="password" id="passInput" class="form-control" placeholder="••••••••" required>
                        <button type="button" onclick="togglePass()" class="pass-eye">👁</button>
                    </div>
                </div>
                <button type="submit" class="btn btn-primary btn-full"><?= __('login_btn') ?></button>
            </form>
            <div class="land-login-footer">
                <p><?= __('no_account') ?> <a href="register.php"><?= __('register') ?></a></p>
                <a href="public/" style="color:rgba(255,255,255,0.3);font-size:0.75rem">🌐 <?= $lang==='en' ? 'Public directory' : 'Directório público' ?></a>
            </div>
        </div>
    </div>
</section>

<!-- ═══ SOBRE ════════════════════════════════════════════════════════ -->
<section class="land-section land-about" id="sobre">
    <div class="land-container">
        <div class="land-section-badge"><?= $lang==='en' ? '✝ About SIGEDEC' : '✝ Sobre o SIGEDEC' ?></div>
        <h2 class="land-h2">
            <?= $lang==='en'
                ? 'A System Born from the Reality of African Churches'
                : 'Um Sistema Nascido da Realidade das Igrejas Africanas' ?>
        </h2>
        <div class="land-about-grid">
            <div class="land-about-text">
                <p><?= $lang==='en'
                    ? 'Africa is home to some of the most vibrant, growing and diverse Christian communities in the world. From the urban megachurches of Lagos and Nairobi to the rural congregations of rural Mozambique, Angola and the Democratic Republic of Congo, African churches are alive with faith, community and purpose.'
                    : 'África é o lar de algumas das comunidades cristãs mais vibrantes, em crescimento e diversas do mundo. Desde as megaergrejas urbanas de Lagos e Nairobi até às congregações rurais de Moçambique, Angola e da República Democrática do Congo, as igrejas africanas pulsam de fé, comunidade e propósito.' ?>
                </p>
                <p><?= $lang==='en'
                    ? 'Yet, managing these communities remains a challenge. Paper records get lost. Excel spreadsheets become unmanageable. Pastors and administrators struggle to track members, finances, and disciplinary records across multiple branches — especially when congregations span entire provinces or countries.'
                    : 'No entanto, gerir estas comunidades continua a ser um desafio. Registos em papel perdem-se. As folhas de cálculo Excel tornam-se ingeríveis. Pastores e administradores têm dificuldade em acompanhar membros, finanças e registos disciplinares em várias filiais — especialmente quando as congregações abrangem províncias inteiras ou países.' ?>
                </p>
                <p><?= $lang==='en'
                    ? 'SIGEDEC was created to solve this problem. Built by Africans, for African churches, it is a complete, secure, mobile-first church management system that respects the specific organisational structures of African denominations — whether you call your branches congregations, conventions, synods, districts, zones or any other term.'
                    : 'O SIGEDEC foi criado para resolver este problema. Desenvolvido por africanos, para igrejas africanas, é um sistema completo, seguro e adaptado a telemóvel que respeita as estruturas organizacionais específicas das denominações africanas — quer chame às suas filiais congregações, convenções, sínodos, distritos, zonas ou qualquer outro termo.' ?>
                </p>
                <p><?= $lang==='en'
                    ? 'Each denomination on the SIGEDEC platform is completely isolated. Your data is your data. No other church, no other admin — not even the platform administrator — can see your members, finances or internal records. Privacy and confidentiality are not optional features: they are the foundation.'
                    : 'Cada denominação na plataforma SIGEDEC é completamente isolada. Os seus dados são os seus dados. Nenhuma outra igreja, nenhum outro administrador — nem mesmo o administrador da plataforma — pode ver os seus membros, finanças ou registos internos. A privacidade e confidencialidade não são funcionalidades opcionais: são a fundação.' ?>
                </p>
            </div>
            <div class="land-about-cards">
                <div class="land-about-card">
                    <div class="land-about-icon">🌍</div>
                    <h3><?= $lang==='en' ? 'Pan-African' : 'Pan-Africano' ?></h3>
                    <p><?= $lang==='en' ? 'Available in Portuguese and English. Designed for any African country.' : 'Disponível em Português e Inglês. Desenhado para qualquer país africano.' ?></p>
                </div>
                <div class="land-about-card">
                    <div class="land-about-icon">📱</div>
                    <h3><?= $lang==='en' ? 'Mobile First' : 'Primeiro Telemóvel' ?></h3>
                    <p><?= $lang==='en' ? 'Works perfectly on any smartphone. No app installation needed.' : 'Funciona perfeitamente em qualquer telemóvel. Sem instalação de aplicação.' ?></p>
                </div>
                <div class="land-about-card">
                    <div class="land-about-icon">🔒</div>
                    <h3><?= $lang==='en' ? 'Data Isolation' : 'Isolamento de Dados' ?></h3>
                    <p><?= $lang==='en' ? 'Each denomination is a closed, private space. No data sharing between churches.' : 'Cada denominação é um espaço fechado e privado. Sem partilha de dados entre igrejas.' ?></p>
                </div>
                <div class="land-about-card">
                    <div class="land-about-icon">💰</div>
                    <h3><?= $lang==='en' ? 'Symbolic Cost' : 'Custo Simbólico' ?></h3>
                    <p><?= $lang==='en' ? 'From $10/year. A fair price to keep the system alive online.' : 'A partir de $10/ano. Um preço justo para manter o sistema na internet.' ?></p>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- ═══ MISSÃO ══════════════════════════════════════════════════════ -->
<section class="land-section land-mission" id="missao">
    <div class="land-container">
        <div class="land-mission-grid">
            <div class="land-mission-item">
                <div class="land-mission-icon">🎯</div>
                <h3><?= $lang==='en' ? 'Our Mission' : 'A Nossa Missão' ?></h3>
                <p><?= $lang==='en'
                    ? 'To empower African churches with digital tools that are accessible, affordable, and designed for their reality — helping pastors and administrators focus on what truly matters: their community.'
                    : 'Capacitar as igrejas africanas com ferramentas digitais acessíveis, acessíveis e desenhadas para a sua realidade — ajudando pastores e administradores a focarem-se no que realmente importa: a sua comunidade.' ?>
                </p>
            </div>
            <div class="land-mission-item">
                <div class="land-mission-icon">👁️</div>
                <h3><?= $lang==='en' ? 'Our Vision' : 'A Nossa Visão' ?></h3>
                <p><?= $lang==='en'
                    ? 'To become the standard church management platform across the African continent, serving denominations from Cape Town to Cairo, from Dakar to Dar es Salaam.'
                    : 'Tornar-mo-nos a plataforma padrão de gestão eclesiástica em todo o continente africano, servindo denominações do Cabo até ao Cairo, de Dacar a Dar es Salaam.' ?>
                </p>
            </div>
            <div class="land-mission-item">
                <div class="land-mission-icon">💎</div>
                <h3><?= $lang==='en' ? 'Our Values' : 'Os Nossos Valores' ?></h3>
                <p><?= $lang==='en'
                    ? 'Transparency, privacy, accessibility and service. We believe technology should serve the church — not complicate it. Simple tools, real results.'
                    : 'Transparência, privacidade, acessibilidade e serviço. Acreditamos que a tecnologia deve servir a igreja — não complicá-la. Ferramentas simples, resultados reais.' ?>
                </p>
            </div>
        </div>
    </div>
</section>

<!-- ═══ FUNCIONALIDADES ══════════════════════════════════════════════ -->
<section class="land-section" id="funcionalidades">
    <div class="land-container">
        <div class="land-section-badge"><?= $lang==='en' ? '⚡ Features' : '⚡ Funcionalidades' ?></div>
        <h2 class="land-h2"><?= $lang==='en' ? 'Everything Your Church Needs' : 'Tudo o que a sua Igreja Precisa' ?></h2>
        <p class="land-section-sub"><?= $lang==='en'
            ? 'A complete, integrated platform so you never need to use spreadsheets or paper records again.'
            : 'Uma plataforma completa e integrada para nunca mais precisar de folhas de cálculo ou registos em papel.' ?>
        </p>
        <div class="land-features-grid">
            <div class="land-feature">
                <div class="land-feat-icon">👥</div>
                <h3><?= $lang==='en' ? 'Member Management' : 'Gestão de Membros' ?></h3>
                <p><?= $lang==='en'
                    ? 'Complete records for each member: personal data, baptism, marital status, dating, marriage, disciplinary history and membership status (active, suspended, unlinked).'
                    : 'Ficha completa de cada membro: dados pessoais, baptismo, estado civil, namoro, casamento, histórico disciplinar e estado de membresia (activo, cortado, desvinculado).' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">🏘️</div>
                <h3><?= $lang==='en' ? 'Flexible Subdivisions' : 'Subdivisões Flexíveis' ?></h3>
                <p><?= $lang==='en'
                    ? 'Create your own denomination structure: congregations, conventions, synods, districts, zones — using exactly the terminology your denomination already uses.'
                    : 'Crie a estrutura da sua denominação: congregações, convenções, sínodos, distritos, zonas — usando exactamente a terminologia que a sua denominação já utiliza.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">✝️</div>
                <h3><?= $lang==='en' ? 'Ministries & Ordination' : 'Ministérios & Consagração' ?></h3>
                <p><?= $lang==='en'
                    ? 'Define ministry roles (bishop, pastor, deacon, elder, evangelist...) and assign ordained members. Full history of ministerial assignments.'
                    : 'Defina cargos ministeriais (bispo, pastor, diácono, ancião, evangelista...) e vincule membros consagrados. Histórico completo de atribuições ministeriais.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">💰</div>
                <h3><?= $lang==='en' ? 'Financial Control' : 'Controlo Financeiro' ?></h3>
                <p><?= $lang==='en'
                    ? 'Record tithes, offerings, donations and expenses. Monthly summaries, balance calculations and printable financial reports.'
                    : 'Registe dízimos, ofertas, doações e despesas. Resumos mensais, cálculo de saldos e relatórios financeiros imprimíveis.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">👑</div>
                <h3><?= $lang==='en' ? 'Leadership Management' : 'Gestão de Liderança' ?></h3>
                <p><?= $lang==='en'
                    ? 'Assign leaders to each subdivision. Full visibility of who leads which congregation, when they started and their specific role.'
                    : 'Atribua líderes a cada subdivisão. Visibilidade total de quem lidera qual congregação, quando começou e qual o seu cargo específico.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">📊</div>
                <h3><?= $lang==='en' ? 'Reports & Statistics' : 'Relatórios & Estatísticas' ?></h3>
                <p><?= $lang==='en'
                    ? 'Printable local and global reports. Statistics on new members, marital status, suspensions, restorations and financial summaries.'
                    : 'Relatórios locais e globais imprimíveis. Estatísticas sobre novos membros, estado civil, cortes, restaurações e resumos financeiros.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">🚫</div>
                <h3><?= $lang==='en' ? 'Blacklist & Restoration' : 'Lista Negra & Restauração' ?></h3>
                <p><?= $lang==='en'
                    ? 'Members removed due to misconduct are stored in a private blacklist — not erased. When they repent and return, restoration is just one click away.'
                    : 'Membros removidos por motivos disciplinares são guardados numa lista negra privada — não apagados. Quando se arrependem e voltam, a restauração é apenas um clique.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">💑</div>
                <h3><?= $lang==='en' ? 'Courtship & Marriage' : 'Namoro & Casamento' ?></h3>
                <p><?= $lang==='en'
                    ? 'Track romantic relationships announced publicly within the church. Record wedding dates and automatically update marital status when the couple gets married.'
                    : 'Acompanhe os relacionamentos anunciados publicamente na igreja. Registe datas de casamento e actualize automaticamente o estado civil quando o casal se casa.' ?>
                </p>
            </div>
            <div class="land-feature">
                <div class="land-feat-icon">🌐</div>
                <h3><?= $lang==='en' ? 'Public Directory' : 'Directório Público' ?></h3>
                <p><?= $lang==='en'
                    ? 'A public page listing all registered denominations, their subdivisions and members with their status (active, suspended, unlinked) visible with colour codes.'
                    : 'Uma página pública que lista todas as denominações registadas, as suas subdivisões e membros com o seu estado (activo, cortado, desvinculado) visível com códigos de cor.' ?>
                </p>
            </div>
        </div>
    </div>
</section>

<!-- ═══ PARA QUEM ════════════════════════════════════════════════════ -->
<section class="land-section land-forwhom" id="paroquem">
    <div class="land-container">
        <div class="land-section-badge"><?= $lang==='en' ? '🙏 Who is it for?' : '🙏 Para quem é?' ?></div>
        <h2 class="land-h2"><?= $lang==='en' ? 'Designed for Every Role in Your Church' : 'Desenhado para Cada Função na sua Igreja' ?></h2>
        <div class="land-roles-grid">
            <div class="land-role">
                <div class="land-role-avatar" style="background:linear-gradient(135deg,#c9a84c,#e2c476)">🏛️</div>
                <h3><?= $lang==='en' ? 'Denomination Administrator' : 'Admin da Denominação' ?></h3>
                <ul>
                    <?php $items_admin = $lang==='en'
                        ? ['Configure subdivisions and hierarchy','Manage ministries and ordinations','View full reports','Manage local users and finance users','See blacklist and restore members']
                        : ['Configurar subdivisões e hierarquia','Gerir ministérios e consagrações','Ver relatórios completos','Gerir utilizadores locais e financeiros','Ver lista negra e restaurar membros'];
                    foreach ($items_admin as $i): ?><li>✓ <?= $i ?></li><?php endforeach; ?>
                </ul>
            </div>
            <div class="land-role">
                <div class="land-role-avatar" style="background:linear-gradient(135deg,#22c55e,#4ade80)">📋</div>
                <h3><?= $lang==='en' ? 'Local Manager' : 'Gestor Local' ?></h3>
                <ul>
                    <?php $items_local = $lang==='en'
                        ? ['Register and manage subdivision members','Record baptisms and civil status','Track courtship and marriage','Update membership status','Generate local reports']
                        : ['Registar e gerir membros da subdivisão','Registar baptismos e estado civil','Acompanhar namoro e casamento','Actualizar estado de membresia','Gerar relatórios locais'];
                    foreach ($items_local as $i): ?><li>✓ <?= $i ?></li><?php endforeach; ?>
                </ul>
            </div>
            <div class="land-role">
                <div class="land-role-avatar" style="background:linear-gradient(135deg,#a855f7,#c084fc)">💰</div>
                <h3><?= $lang==='en' ? 'Finance Manager' : 'Gestor Financeiro' ?></h3>
                <ul>
                    <?php $items_fin = $lang==='en'
                        ? ['Create income categories (tithes, offerings...)','Record all income and expenses','Filter by month and category','View financial summaries','Print financial reports']
                        : ['Criar categorias de entrada (dízimos, ofertas...)','Registar todas as entradas e saídas','Filtrar por mês e categoria','Ver resumos financeiros','Imprimir relatórios financeiros'];
                    foreach ($items_fin as $i): ?><li>✓ <?= $i ?></li><?php endforeach; ?>
                </ul>
            </div>
        </div>
    </div>
</section>

<!-- ═══ TUTORIAL ════════════════════════════════════════════════════ -->
<section class="land-section" id="tutorial">
    <div class="land-container">
        <div class="land-section-badge"><?= $lang==='en' ? '📖 Tutorial' : '📖 Tutorial' ?></div>
        <h2 class="land-h2"><?= $lang==='en' ? 'How to Get Started in 4 Steps' : 'Como Começar em 4 Passos' ?></h2>
        <div class="land-steps">
            <div class="land-step">
                <div class="land-step-num">1</div>
                <div class="land-step-content">
                    <h3><?= $lang==='en' ? 'Register your denomination' : 'Registe a sua denominação' ?></h3>
                    <p><?= $lang==='en'
                        ? 'Click "Register Denomination" and fill in the name of your church, your country, province, district and administrator credentials. Your 6-month free trial starts automatically. No payment required.'
                        : 'Clique em "Registar Denominação" e preencha o nome da sua igreja, o seu país, província, distrito e as credenciais do administrador. O período de teste gratuito de 6 meses começa automaticamente. Sem pagamento.' ?>
                    </p>
                </div>
            </div>
            <div class="land-step">
                <div class="land-step-num">2</div>
                <div class="land-step-content">
                    <h3><?= $lang==='en' ? 'Set up your structure' : 'Configure a sua estrutura' ?></h3>
                    <p><?= $lang==='en'
                        ? 'Log in as denomination administrator. Go to "Denominations & Subdivisions" and create your church structure. Add your congregations, conventions or districts. Define the terms your denomination uses (synod, convention, congregation...).'
                        : 'Entre como administrador da denominação. Vá a "Denominações & Subdivisões" e crie a estrutura da sua igreja. Adicione as suas congregações, convenções ou distritos. Defina os termos que a sua denominação usa (sínodo, convenção, congregação...).' ?>
                    </p>
                </div>
            </div>
            <div class="land-step">
                <div class="land-step-num">3</div>
                <div class="land-step-content">
                    <h3><?= $lang==='en' ? 'Create local users' : 'Crie utilizadores locais' ?></h3>
                    <p><?= $lang==='en'
                        ? 'In "Users", create accounts for local managers (one per subdivision) and finance managers. Each local manager will only see and manage the members of their own subdivision — guaranteeing data security even within the same denomination.'
                        : 'Em "Utilizadores", crie contas para os gestores locais (um por subdivisão) e gestores financeiros. Cada gestor local apenas verá e gerirá os membros da sua própria subdivisão — garantindo segurança de dados mesmo dentro da mesma denominação.' ?>
                    </p>
                </div>
            </div>
            <div class="land-step">
                <div class="land-step-num">4</div>
                <div class="land-step-content">
                    <h3><?= $lang==='en' ? 'Start registering members' : 'Comece a registar membros' ?></h3>
                    <p><?= $lang==='en'
                        ? 'Go to "Members" and add the first members of each subdivision. Include their baptism date, personal details, civil status and membership status. The system is ready to generate statistics and reports immediately.'
                        : 'Vá a "Membros" e adicione os primeiros membros de cada subdivisão. Inclua a data de baptismo, dados pessoais, estado civil e estado de membresia. O sistema já está pronto para gerar estatísticas e relatórios de imediato.' ?>
                    </p>
                </div>
            </div>
        </div>

        <!-- Sub-tutorial: Recursos avançados -->
        <div class="land-advanced-tips">
            <h3><?= $lang==='en' ? 'Advanced Features' : 'Funcionalidades Avançadas' ?></h3>
            <div class="land-tips-grid">
                <div class="land-tip">
                    <strong>🔶 <?= $lang==='en' ? 'Suspensions & Restorations' : 'Cortes & Restaurações' ?></strong>
                    <p><?= $lang==='en'
                        ? 'When changing a member\'s status to "Suspended", the system records the reason and circumstances. Each suspension is counted. When restored, the cycle is recorded. You always have a complete history.'
                        : 'Ao alterar o estado de um membro para "Cortado", o sistema regista o motivo e as circunstâncias. Cada corte é contabilizado. Ao ser devolvido, o ciclo fica registado. Tem sempre um histórico completo.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>💑 <?= $lang==='en' ? 'Courtship Tracking' : 'Acompanhamento de Namoro' ?></strong>
                    <p><?= $lang==='en'
                        ? 'When a single member is in a relationship, mark their partner\'s name and the public announcement date. The system shows "Brother/Sister [Name] dating [Name]". On the wedding date, both are automatically updated to married status.'
                        : 'Quando um membro solteiro está em namoro, marque o nome do parceiro e a data de apresentação pública. O sistema mostra "Irmão/ã [Nome] em namoro com [Nome]". Na data do casamento, ambos são actualizados automaticamente para casados.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>🌐 <?= $lang==='en' ? 'Denomination Language' : 'Idioma da Denominação' ?></strong>
                    <p><?= $lang==='en'
                        ? 'The system supports Portuguese and English. The language can be changed at any time from the dashboard corner. The change only affects visual text — data is preserved.'
                        : 'O sistema suporta Português e Inglês. O idioma pode ser alterado a qualquer momento no canto do painel. A mudança apenas afecta o texto visual — os dados são preservados.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>📊 <?= $lang==='en' ? 'Reports & PDF' : 'Relatórios e PDF' ?></strong>
                    <p><?= $lang==='en'
                        ? 'All reports include a "Print / PDF" button. Use your browser\'s print function (Ctrl+P) and choose "Save as PDF" to generate a file you can share with church leadership or archive.'
                        : 'Todos os relatórios incluem um botão "Imprimir / PDF". Use a função de impressão do seu browser (Ctrl+P) e escolha "Guardar como PDF" para gerar um ficheiro que pode partilhar com a liderança da igreja ou arquivar.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>🙏 <?= $lang==='en' ? 'Member Portal' : 'Portal do Membro' ?></strong>
                    <p><?= $lang==='en'
                        ? 'Each member can have their own secure login. The denomination admin or local pastor assigns a portal email and password. Once logged in, the member sees their complete profile — name, baptism, marital status, membership status, and tithe classification — but can only change their own photo (max 2 MB) and password. All other data remains read-only.'
                        : 'Cada membro pode ter o seu próprio acesso seguro. O admin da denominação ou o pastor local atribui ao membro um email e senha de portal. Uma vez dentro, o membro vê a sua ficha completa — nome, baptismo, estado civil, estado de membresia e situação do dízimo — mas só pode alterar a própria foto (máx. 2 MB) e a senha. Todos os outros dados ficam só de leitura.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>💰 <?= $lang==='en' ? 'Tithe / Contribution Status' : 'Situação do Dízimo / Contribuições' ?></strong>
                    <p><?= $lang==='en'
                        ? 'The finance manager can classify each member\'s contribution behaviour as: Regular (consistent), Irregular (oscillating), or Unfaithful (stopped over 3 months). This classification is visible on the member\'s portal and helps leadership have an accurate pastoral picture without exposing private financial amounts.'
                        : 'O gestor financeiro pode classificar o comportamento de cada membro quanto ao dízimo: Regular (consistente), Irregular (oscilante) ou Infiel (parou há mais de 3 meses). Esta classificação é visível no portal do membro e ajuda a liderança a ter um quadro pastoral rigoroso sem expor valores financeiros.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>✂️ <?= $lang==='en' ? 'Communion Cut & Unlinking Authority' : 'Corte de Comunhão e Desvinculação' ?></strong>
                    <p><?= $lang==='en'
                        ? 'The authority to suspend a member from communion or formally unlink them belongs to the local pastor or leader. In the member\'s edit form, the pastor selects Suspended or Unlinked, enters the reason, and saves. The system logs the action automatically with date and reason.'
                        : 'A autoridade para cortar um membro da comunhão ou desvinculá-lo pertence ao pastor/líder local — o Gestor Local ao nível da congregação. Na ficha de edição do membro, o pastor selecciona o estado, insere o motivo e grava. O sistema regista a acção automaticamente.' ?>
                    </p>
                </div>
                <div class="land-tip">
                    <strong>💾 <?= $lang==='en' ? 'Backup & Data Security' : 'Cópia de Segurança' ?></strong>
                    <p><?= $lang==='en'
                        ? 'Denomination administrators can download a complete backup of their data at any time — members, photos, finances, subdivisions — as a single ZIP file. The backup page shows the date of the last backup. The general system administrator can download a full backup of all denominations.'
                        : 'Os administradores de denominação podem descarregar uma cópia completa dos seus dados a qualquer momento — membros, fotos, finanças, subdivisões — como um único ficheiro ZIP. A página de backup mostra a data da última cópia efectuada. O administrador geral pode descarregar um backup completo de todas as denominações.' ?>
                    </p>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- ═══ AFRICA SECTION ══════════════════════════════════════════════ -->
<section class="land-section land-africa" id="africa">
    <div class="land-container">
        <div class="land-section-badge">🌍 Africa</div>
        <h2 class="land-h2"><?= $lang==='en' ? 'Why Africa Needs Its Own Church System' : 'Por que a África Precisa do seu Próprio Sistema Eclesiástico' ?></h2>
        <div class="land-africa-text">
            <p><?= $lang==='en'
                ? 'Africa is the continent with the fastest-growing Christian population on the planet. According to various studies, more than 600 million Christians live in Africa — and this number continues to grow every year. In countries like Mozambique, Angola, Nigeria, Ghana, Kenya, Uganda, the Democratic Republic of Congo and dozens of others, the church is not just a place of worship: it is a pillar of the community.'
                : 'África é o continente com a população cristã de crescimento mais rápido do planeta. Segundo vários estudos, mais de 600 milhões de cristãos vivem em África — e este número continua a crescer todos os anos. Em países como Moçambique, Angola, Nigéria, Gana, Quénia, Uganda, a República Democrática do Congo e dezenas de outros, a igreja não é apenas um local de culto: é um pilar da comunidade.' ?>
            </p>
            <p><?= $lang==='en'
                ? 'However, the technological tools available for church management were largely developed for Western churches — with structures, terminology and realities that do not match the African context. The result is that many African denominations end up using generic tools that do not adapt to their specific reality, or worse, continue managing everything manually.'
                : 'No entanto, as ferramentas tecnológicas disponíveis para gestão eclesiástica foram maioritariamente desenvolvidas para igrejas ocidentais — com estruturas, terminologia e realidades que não correspondem ao contexto africano. O resultado é que muitas denominações africanas acabam por usar ferramentas genéricas que não se adaptam à sua realidade específica, ou pior, continuam a gerir tudo manualmente.' ?>
            </p>
            <p><?= $lang==='en'
                ? 'SIGEDEC changes this. It is a system designed with African denominations in mind from the very first line of code. The flexible terminology system (congregation, convention, synod, district...) allows any denomination, regardless of its internal organisational structure, to use the system naturally without having to adapt its own language.'
                : 'O SIGEDEC muda isso. É um sistema desenhado com as denominações africanas em mente desde a primeira linha de código. O sistema de terminologia flexível (congregação, convenção, sínodo, distrito...) permite que qualquer denominação, independentemente da sua estrutura organizacional interna, use o sistema de forma natural sem ter de adaptar a sua própria linguagem.' ?>
            </p>
            <p><?= $lang==='en'
                ? 'The pricing also reflects this reality. At just $10 per year, SIGEDEC is accessible even to the smallest denominations in the most economically challenged regions of the continent. Our goal is not profit: it is sustainability. We charge only enough to keep the system running online and to continue improving it.'
                : 'Os preços também reflectem esta realidade. Com apenas 10 dólares por ano, o SIGEDEC é acessível mesmo às denominações mais pequenas nas regiões economicamente mais desafiantes do continente. O nosso objectivo não é o lucro: é a sustentabilidade. Cobramos apenas o suficiente para manter o sistema a funcionar online e continuar a melhorá-lo.' ?>
            </p>
            <p><?= $lang==='en'
                ? 'Although SIGEDEC was designed for Africa, it is not exclusively for Africa. Any church or denomination anywhere in the world can use the platform. But our heart, our focus and our commitment are for the African church — vibrant, diverse, growing and deserving of the best digital tools.'
                : 'Embora o SIGEDEC tenha sido desenhado para África, não é exclusivo para África. Qualquer igreja ou denominação em qualquer parte do mundo pode usar a plataforma. Mas o nosso coração, o nosso foco e o nosso comprometimento são para a igreja africana — vibrante, diversa, em crescimento e merecedora das melhores ferramentas digitais.' ?>
            </p>
        </div>
    </div>
</section>

<!-- ═══ PREÇOS ═══════════════════════════════════════════════════════ -->
<section class="land-section" id="precos">
    <div class="land-container">
        <div class="land-section-badge">💰 <?= $lang==='en' ? 'Pricing' : 'Preços' ?></div>
        <h2 class="land-h2"><?= $lang==='en' ? 'Simple, Fair, African Pricing' : 'Preços Simples, Justos e Africanos' ?></h2>
        <p class="land-section-sub"><?= $lang==='en'
            ? 'Start for free. Pay only if you want to continue after 6 months — at a price that makes sense for Africa.'
            : 'Comece gratuitamente. Pague apenas se quiser continuar após 6 meses — a um preço que faz sentido para África.' ?>
        </p>
        <div class="land-pricing-grid">
            <div class="land-price-card">
                <div class="land-price-header">
                    <div class="land-price-badge"><?= $lang==='en' ? 'Free Trial' : 'Período de Teste' ?></div>
                    <div class="land-price-value">$0</div>
                    <div class="land-price-period"><?= $lang==='en' ? '6 months' : '6 meses' ?></div>
                </div>
                <ul class="land-price-features">
                    <li>✓ <?= $lang==='en' ? 'All features included' : 'Todas as funcionalidades' ?></li>
                    <li>✓ <?= $lang==='en' ? 'Unlimited members' : 'Membros ilimitados' ?></li>
                    <li>✓ <?= $lang==='en' ? 'No credit card required' : 'Sem cartão de crédito' ?></li>
                    <li>✓ <?= $lang==='en' ? 'Automatic on registration' : 'Automático ao registar' ?></li>
                </ul>
                <a href="register.php" class="btn btn-outline btn-full"><?= __('reg_btn') ?></a>
            </div>
            <div class="land-price-card land-price-featured">
                <div class="land-price-popular"><?= $lang==='en' ? 'Most popular' : 'Mais popular' ?></div>
                <div class="land-price-header">
                    <div class="land-price-badge">1 <?= $lang==='en' ? 'Year' : 'Ano' ?></div>
                    <div class="land-price-value">$10</div>
                    <div class="land-price-period">/ <?= $lang==='en' ? 'year' : 'ano' ?></div>
                </div>
                <ul class="land-price-features">
                    <li>✓ <?= $lang==='en' ? 'All features' : 'Todas as funcionalidades' ?></li>
                    <li>✓ <?= $lang==='en' ? '12 months access' : '12 meses de acesso' ?></li>
                    <li>✓ <?= $lang==='en' ? 'Priority support' : 'Suporte prioritário' ?></li>
                    <li>✓ <?= $lang==='en' ? 'License key via admin' : 'Chave de licença via admin' ?></li>
                </ul>
                <a href="mailto:<?= CONTACT_EMAIL ?>?subject=Licença SIGEDEC 1 Ano" class="btn btn-primary btn-full"><?= $lang==='en' ? 'Get License' : 'Obter Licença' ?></a>
            </div>
            <div class="land-price-card">
                <div class="land-price-header">
                    <div class="land-price-badge">2 <?= $lang==='en' ? 'Years' : 'Anos' ?></div>
                    <div class="land-price-value">$15</div>
                    <div class="land-price-period">/ 2 <?= $lang==='en' ? 'years' : 'anos' ?></div>
                </div>
                <ul class="land-price-features">
                    <li>✓ <?= $lang==='en' ? 'All features' : 'Todas as funcionalidades' ?></li>
                    <li>✓ <?= $lang==='en' ? '24 months access' : '24 meses de acesso' ?></li>
                    <li>✓ <?= $lang==='en' ? 'Save $5 vs annual' : 'Poupa $5 vs anual' ?></li>
                    <li>✓ <?= $lang==='en' ? 'License key via admin' : 'Chave de licença via admin' ?></li>
                </ul>
                <a href="mailto:<?= CONTACT_EMAIL ?>?subject=Licença SIGEDEC 2 Anos" class="btn btn-outline btn-full"><?= $lang==='en' ? 'Get License' : 'Obter Licença' ?></a>
            </div>
            <div class="land-price-card">
                <div class="land-price-header">
                    <div class="land-price-badge">3 <?= $lang==='en' ? 'Years' : 'Anos' ?></div>
                    <div class="land-price-value">$20</div>
                    <div class="land-price-period">/ 3 <?= $lang==='en' ? 'years' : 'anos' ?></div>
                </div>
                <ul class="land-price-features">
                    <li>✓ <?= $lang==='en' ? 'All features' : 'Todas as funcionalidades' ?></li>
                    <li>✓ <?= $lang==='en' ? '36 months access' : '36 meses de acesso' ?></li>
                    <li>✓ <?= $lang==='en' ? 'Best value' : 'Melhor valor' ?></li>
                    <li>✓ <?= $lang==='en' ? 'License key via admin' : 'Chave de licença via admin' ?></li>
                </ul>
                <a href="mailto:<?= CONTACT_EMAIL ?>?subject=Licença SIGEDEC 3 Anos" class="btn btn-outline btn-full"><?= $lang==='en' ? 'Get License' : 'Obter Licença' ?></a>
            </div>
        </div>
        <p class="land-price-note">
            * <?= $lang==='en'
                ? 'Prices in USD for easier continental transactions. Payment methods are arranged directly with the administrator: '
                : 'Preços em dólares para facilitar transacções continentais. Os métodos de pagamento são combinados directamente com o administrador: ' ?>
            <a href="mailto:<?= CONTACT_EMAIL ?>"><?= CONTACT_EMAIL ?></a>
        </p>
    </div>
</section>

<!-- ═══ CONTACTO ════════════════════════════════════════════════════ -->
<!-- ═══ STANDALONE EDITION ══════════════════════════════════ -->
<section class="land-section land-standalone" id="standalone">
    <div class="land-container">

        <div class="land-section-badge">📦 <?= $lang==='en' ? 'Standalone Edition' : 'Edição Autónoma' ?></div>
        <h2 class="land-h2"><?= $lang==='en'
            ? 'Install SIGEDEC on Your Own Server'
            : 'Instale o SIGEDEC no Servidor da Sua Denominação' ?>
        </h2>
        <p class="land-section-sub"><?= $lang==='en'
            ? 'Does your denomination want full control over its own data? The Standalone Edition lets you download SIGEDEC and install it directly on your own hosting — just like WordPress, but built specifically for African churches.'
            : 'A sua denominação quer controlo total sobre os seus próprios dados? A Edição Autónoma permite-lhe descarregar o SIGEDEC e instalá-lo directamente no seu próprio servidor — tal como o WordPress, mas desenhado especificamente para igrejas africanas.' ?>
        </p>

        <!-- Comparação: Cloud vs Standalone -->
        <div class="standalone-compare">
            <div class="standalone-compare-card standalone-compare-cloud">
                <div class="standalone-compare-icon">☁️</div>
                <h3><?= $lang==='en' ? 'Cloud Version' : 'Versão Cloud' ?></h3>
                <p><?= $lang==='en'
                    ? 'Hosted at sigedec.asaf.fyi. Register and start in 2 minutes. Ideal for most denominations.'
                    : 'Alojada em sigedec.asaf.fyi. Registe-se e comece em 2 minutos. Ideal para a maioria das denominações.' ?>
                </p>
                <ul>
                    <li>✅ <?= $lang==='en' ? '6 months free trial' : '6 meses de teste gratuito' ?></li>
                    <li>✅ <?= $lang==='en' ? 'From $10/year' : 'A partir de $10/ano' ?></li>
                    <li>✅ <?= $lang==='en' ? 'No installation needed' : 'Sem instalação' ?></li>
                    <li>✅ <?= $lang==='en' ? 'Automatic updates' : 'Actualizações automáticas' ?></li>
                    <li>☁️ <?= $lang==='en' ? 'Data hosted on our servers' : 'Dados no nosso servidor' ?></li>
                </ul>
                <a href="register.php" class="btn btn-outline" style="margin-top:1rem"><?= $lang==='en' ? 'Try Free →' : 'Experimentar Grátis →' ?></a>
            </div>
            <div class="standalone-compare-divider">
                <span><?= $lang==='en' ? 'OR' : 'OU' ?></span>
            </div>
            <div class="standalone-compare-card standalone-compare-standalone">
                <div class="standalone-compare-icon">🏛️</div>
                <h3><?= $lang==='en' ? 'Standalone Edition' : 'Edição Autónoma' ?></h3>
                <p><?= $lang==='en'
                    ? 'Download and install on your own server. Your data, your domain, your control. A one-time payment.'
                    : 'Descarregue e instale no seu próprio servidor. Os seus dados, o seu domínio, o seu controlo. Pagamento único.' ?>
                </p>
                <ul>
                    <li>✅ <?= $lang==='en' ? 'All features included' : 'Todas as funcionalidades incluídas' ?></li>
                    <li>✅ <?= $lang==='en' ? 'Your own domain (e.g. gestao.suaigreja.com)' : 'O seu próprio domínio (ex: gestao.suaigreja.com)' ?></li>
                    <li>✅ <?= $lang==='en' ? '<strong>$20 one-time payment, forever</strong>' : '<strong>$20 pagamento único, para sempre</strong>' ?></li>
                    <li>✅ <?= $lang==='en' ? 'PT and EN included' : 'PT e EN incluídos' ?></li>
                    <li>🔒 <?= $lang==='en' ? 'Requires activation key (anti-piracy)' : 'Requer chave de activação (anti-pirataria)' ?></li>
                </ul>
                <a href="/downloads/download.php" class="btn btn-primary" style="margin-top:1rem" download>
                    ⬇️ <?= $lang==='en' ? 'Download Standalone — Free' : 'Descarregar Standalone — Grátis' ?>
                </a>
            </div>
        </div>

        <!-- Download CTA -->
        <div class="standalone-download-box" id="standalone-download">
            <div class="standalone-dl-left">
                <div class="standalone-dl-icon">📦</div>
                <div>
                    <div class="standalone-dl-title">SIGEDEC Standalone Edition</div>
                    <div class="standalone-dl-meta">
                        PHP 8.0+ · <?= $lang==='en' ? 'Any shared hosting · ~70KB ZIP' : 'Qualquer hosting partilhado · ~70KB ZIP' ?>
                    </div>
                </div>
            </div>
            <div class="standalone-dl-right">
                <div class="standalone-price-tag">
                    <span class="standalone-price">$20</span>
                    <span class="standalone-price-sub"><?= $lang==='en' ? 'one-time · forever' : 'único · para sempre' ?></span>
                </div>
                <div style="display:flex;flex-direction:column;gap:.6rem;align-items:flex-end">
                    <a href="/downloads/download.php" class="btn btn-primary btn-lg" download>
                        ⬇️ <?= $lang==='en' ? 'Download Free' : 'Descarregar Grátis' ?>
                    </a>
                    <a href="mailto:<?= CONTACT_EMAIL ?>?subject=<?= urlencode($lang==='en' ? 'SIGEDEC Standalone — Activation Key' : 'SIGEDEC Standalone — Chave de Activação') ?>" class="btn btn-outline btn-sm" style="border-color:rgba(255,255,255,.25);color:rgba(255,255,255,.65)">
                        🔑 <?= $lang==='en' ? 'Get Activation Key ($20)' : 'Obter Chave de Activação ($20)' ?>
                    </a>
                </div>
            </div>
        </div>
        <p style="text-align:center;font-size:.8rem;color:rgba(255,255,255,.45);margin-top:.75rem">
            <?= $lang==='en'
                ? '⬇️ Download is free. The system only works after entering an activation key ($20, one-time). Contact audivido3@gmail.com after downloading.'
                : '⬇️ O download é gratuito. O sistema só funciona após inserir uma chave de activação ($20, pagamento único). Contacte audivido3@gmail.com após descarregar.' ?>
        </p>

        <!-- Como funciona -->
        <div class="standalone-how">
            <h3 class="standalone-how-title">
                <?= $lang==='en' ? 'How does it work, step by step?' : 'Como funciona, passo a passo?' ?>
            </h3>
            <div class="standalone-steps">
                <div class="standalone-step">
                    <div class="standalone-step-num">1</div>
                    <div class="standalone-step-icon">✉️</div>
                    <div class="standalone-step-text">
                        <strong><?= $lang==='en' ? 'Contact' : 'Contactar' ?></strong>
                        <p><?= $lang==='en'
                            ? 'Send an email to audivido3@gmail.com requesting the Standalone Edition. Describe your denomination and your domain (e.g. gestao.suaigreja.com).'
                            : 'Envie um email para audivido3@gmail.com a solicitar a Edição Autónoma. Descreva a sua denominação e o domínio onde vai instalar (ex: gestao.suaigreja.com).' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-step">
                    <div class="standalone-step-num">2</div>
                    <div class="standalone-step-icon">💰</div>
                    <div class="standalone-step-text">
                        <strong><?= $lang==='en' ? 'Pay once' : 'Pagar uma vez' ?></strong>
                        <p><?= $lang==='en'
                            ? '$20 USD — a one-time, permanent payment. No monthly fees, no renewals, no expiry date. Choose the payment method available in your country: M-Pesa, Airtel Money, Orange Money, bank transfer or Western Union.'
                            : '$20 USD — pagamento único e definitivo. Sem mensalidades, sem renovações, sem data de validade. Escolha o método disponível no seu país: M-Pesa, Airtel Money, Orange Money, transferência bancária ou Western Union.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-step">
                    <div class="standalone-step-num">3</div>
                    <div class="standalone-step-icon">📬</div>
                    <div class="standalone-step-text">
                        <strong><?= $lang==='en' ? 'Receive the files' : 'Receber os ficheiros' ?></strong>
                        <p><?= $lang==='en'
                            ? 'After payment confirmation, you receive by email: the download link for the ZIP file (~70KB) and your unique activation key in the format standalone-XXXX-XXXX-XXXX-XXXX. This key is permanently linked to your domain.'
                            : 'Após confirmação do pagamento, recebe por email: o link de download do ficheiro ZIP (~70KB) e a sua chave de activação única no formato standalone-XXXX-XXXX-XXXX-XXXX. Esta chave fica permanentemente ligada ao seu domínio.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-step">
                    <div class="standalone-step-num">4</div>
                    <div class="standalone-step-icon">📤</div>
                    <div class="standalone-step-text">
                        <strong><?= $lang==='en' ? 'Upload to your server' : 'Carregar para o servidor' ?></strong>
                        <p><?= $lang==='en'
                            ? 'Extract the ZIP and upload all files to your hosting via FTP or cPanel File Manager — to the public_html, www or htdocs folder, just like any website. Works on any shared hosting that supports PHP 8.0.'
                            : 'Extraia o ZIP e faça upload de todos os ficheiros para o seu hosting via FTP ou Gestor de Ficheiros do cPanel — para a pasta public_html, www ou htdocs, como qualquer site normal. Funciona em qualquer hosting partilhado com PHP 8.0.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-step">
                    <div class="standalone-step-num">5</div>
                    <div class="standalone-step-icon">🔑</div>
                    <div class="standalone-step-text">
                        <strong><?= $lang==='en' ? 'Activate' : 'Activar' ?></strong>
                        <p><?= $lang==='en'
                            ? 'Open the browser and go to your domain. You will be automatically redirected to the activation page. Paste the key you received and click "Activate". The system validates the key against our server in real time and links it to your domain. This is a one-time step.'
                            : 'Abra o browser e aceda ao seu domínio. Será automaticamente redirecionado para a página de activação. Cole a chave que recebeu e clique em "Activar". O sistema valida a chave contra o nosso servidor em tempo real e associa-a ao seu domínio. Este passo é feito apenas uma vez.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-step">
                    <div class="standalone-step-num">6</div>
                    <div class="standalone-step-icon">⚙️</div>
                    <div class="standalone-step-text">
                        <strong><?= $lang==='en' ? 'Configure and use' : 'Configurar e usar' ?></strong>
                        <p><?= $lang==='en'
                            ? 'After activation, a simple wizard asks for the denomination name, country and administrator credentials. Click "Complete Installation" and the system is ready. Log in and start using it immediately — no technical knowledge required.'
                            : 'Após a activação, um wizard simples pede o nome da denominação, país e credenciais do administrador. Clique em "Concluir Instalação" e o sistema está pronto. Faça login e comece a usar imediatamente — sem necessidade de conhecimentos técnicos.' ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Requisitos Técnicos -->
        <div class="standalone-tech">
            <h3 class="standalone-how-title">
                🛠️ <?= $lang==='en' ? 'Technical Requirements' : 'Requisitos Técnicos' ?>
            </h3>
            <div class="standalone-tech-grid">
                <div class="standalone-tech-item">
                    <span class="standalone-tech-icon">🐘</span>
                    <div>
                        <strong>PHP 8.0+</strong>
                        <p><?= $lang==='en'
                            ? 'Available on virtually all hostings worldwide. Most African providers (Afrihost, Web4Africa, TechHosting) already include it.'
                            : 'Disponível em praticamente todos os hostings mundiais. A maioria dos providers africanos (Afrihost, Web4Africa, TechHosting) já o incluem.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-tech-item">
                    <span class="standalone-tech-icon">🌐</span>
                    <div>
                        <strong><?= $lang==='en' ? 'Web Server' : 'Servidor Web' ?></strong>
                        <p><?= $lang==='en'
                            ? 'Apache, Nginx or LiteSpeed. All major hostings already have this configured.'
                            : 'Apache, Nginx ou LiteSpeed. Todos os hostings principais já têm isto configurado.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-tech-item">
                    <span class="standalone-tech-icon">💾</span>
                    <div>
                        <strong><?= $lang==='en' ? 'No database' : 'Sem base de dados' ?></strong>
                        <p><?= $lang==='en'
                            ? 'SIGEDEC uses JSON files to store data. No MySQL configuration required. Works on the most basic shared hosting plans.'
                            : 'O SIGEDEC usa ficheiros JSON para guardar dados. Não é necessária nenhuma configuração de MySQL. Funciona nos planos de hosting partilhado mais básicos.' ?>
                        </p>
                    </div>
                </div>
                <div class="standalone-tech-item">
                    <span class="standalone-tech-icon">📡</span>
                    <div>
                        <strong><?= $lang==='en' ? 'Internet connection' : 'Ligação à internet' ?></strong>
                        <p><?= $lang==='en'
                            ? 'Required for initial activation and periodic license verification (every 7 days). Daily use does not require permanent connection.'
                            : 'Necessária para a activação inicial e verificação periódica da licença (a cada 7 dias). O uso diário não requer ligação permanente.' ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Anti-pirataria explicado -->
        <div class="standalone-security">
            <div class="standalone-security-icon">🔒</div>
            <div class="standalone-security-text">
                <h4><?= $lang==='en' ? 'How is anti-piracy protection guaranteed?' : 'Como é garantida a protecção anti-pirataria?' ?></h4>
                <p><?= $lang==='en'
                    ? 'The activation key is validated in real time against our server the moment it is used. The key is permanently linked to the domain where it was activated. If someone copies the files to another domain and tries to use the same key, the server responds immediately: "Key already activated on domain X." There is no way to circumvent this without access to our main server. Additionally, the system verifies the license silently every 7 days. If it cannot reach the verification server for more than 10 consecutive days, access is suspended until the next successful verification. This means every installation is traceable and unambiguously linked to the denomination that paid for it.'
                    : 'A chave de activação é validada em tempo real contra o nosso servidor no momento em que é utilizada. A chave fica permanentemente associada ao domínio onde foi activada. Se alguém copiar os ficheiros para outro domínio e tentar usar a mesma chave, o servidor responde de imediato: "Chave já activada no domínio X." Não há forma de contornar isto sem acesso ao nosso servidor principal. Adicionalmente, o sistema verifica a licença silenciosamente a cada 7 dias. Se não conseguir contactar o servidor de verificação durante mais de 10 dias consecutivos, o acesso é suspenso até à próxima verificação bem-sucedida. Isto significa que cada instalação é rastreável e inequivocamente ligada à denominação que a adquiriu.' ?>
                </p>
            </div>
        </div>

        <!-- FAQ standalone -->
        <div class="standalone-faq">
            <h3 class="standalone-how-title">
                ❓ <?= $lang==='en' ? 'Questions about the Standalone Edition' : 'Dúvidas sobre a Edição Autónoma' ?>
            </h3>
            <div class="faq-grid" style="margin-top:1.25rem">
                <div class="faq-block">
                    <div class="faq-item">
                        <button class="faq-q" onclick="toggleFaq(this)">
                            <?= $lang==='en'
                                ? 'Can I install it on a subdomain?'
                                : 'Posso instalar numa subpasta ou subdomínio?' ?>
                            <span class="faq-arrow">▾</span>
                        </button>
                        <div class="faq-a">
                            <p><?= $lang==='en'
                                ? 'Yes. You can install at the root (suaigreja.com), on a subdomain (gestao.suaigreja.com) or in a subfolder (suaigreja.com/gestao). The activation key is linked to the host (the domain or subdomain), so choose where you want to install permanently before activating.'
                                : 'Sim. Pode instalar na raiz (suaigreja.com), num subdomínio (gestao.suaigreja.com) ou numa subpasta (suaigreja.com/gestao). A chave de activação fica ligada ao host (o domínio ou subdomínio), por isso escolha onde quer instalar de forma definitiva antes de activar.' ?>
                            </p>
                        </div>
                    </div>
                    <div class="faq-item">
                        <button class="faq-q" onclick="toggleFaq(this)">
                            <?= $lang==='en'
                                ? 'Does the $20 include future updates?'
                                : 'Os $20 incluem actualizações futuras?' ?>
                            <span class="faq-arrow">▾</span>
                        </button>
                        <div class="faq-a">
                            <p><?= $lang==='en'
                                ? 'The $20 gives you the current version permanently. Major future updates (new modules, new features) may be available as paid upgrades at a reduced price. Minor bug fixes are free and we communicate them by email.'
                                : 'Os $20 dão-lhe a versão actual de forma permanente. Actualizações futuras significativas (novos módulos, novas funcionalidades) poderão estar disponíveis como upgrades pagos a preço reduzido. Correcções de erros menores são gratuitas e comunicamos por email.' ?>
                            </p>
                        </div>
                    </div>
                    <div class="faq-item">
                        <button class="faq-q" onclick="toggleFaq(this)">
                            <?= $lang==='en'
                                ? 'Can I transfer the installation to another domain later?'
                                : 'Posso transferir a instalação para outro domínio mais tarde?' ?>
                            <span class="faq-arrow">▾</span>
                        </button>
                        <div class="faq-a">
                            <p><?= $lang==='en'
                                ? 'Yes, but this requires contacting us. We unlock the key from the old domain and you reactivate it on the new one. This service is free for legitimate domain migrations (e.g. if you change your church website). Abuse of this process is monitored.'
                                : 'Sim, mas requer que nos contacte. Desbloqueamos a chave do domínio antigo e volta a activar no novo. Este serviço é gratuito para migrações legítimas de domínio (ex: se mudar o site da igreja). O abuso deste processo é monitorado.' ?>
                            </p>
                        </div>
                    </div>
                </div>
                <div class="faq-block">
                    <div class="faq-item">
                        <button class="faq-q" onclick="toggleFaq(this)">
                            <?= $lang==='en'
                                ? 'What happens if I lose my admin password?'
                                : 'O que acontece se perder a senha do administrador?' ?>
                            <span class="faq-arrow">▾</span>
                        </button>
                        <div class="faq-a">
                            <p><?= $lang==='en'
                                ? 'Contact us at audivido3@gmail.com. We provide you with a recovery procedure via direct file access on your server (via FTP/cPanel). We cannot recover passwords remotely — but we can guide you step by step to create a new administrator account.'
                                : 'Contacte-nos em audivido3@gmail.com. Fornecemos um procedimento de recuperação via acesso directo aos ficheiros no seu servidor (via FTP/cPanel). Não conseguimos recuperar senhas remotamente — mas guiamos passo a passo para criar uma nova conta de administrador.' ?>
                            </p>
                        </div>
                    </div>
                    <div class="faq-item">
                        <button class="faq-q" onclick="toggleFaq(this)">
                            <?= $lang==='en'
                                ? 'Is the Standalone different from the cloud version?'
                                : 'A versão Standalone é diferente da versão cloud?' ?>
                            <span class="faq-arrow">▾</span>
                        </button>
                        <div class="faq-a">
                            <p><?= $lang==='en'
                                ? 'Functionally they are identical. The Standalone does not have the public directory (which lists all denominations registered in the cloud), the registration page, or the landing page — because none of that makes sense on a private installation. Everything else is the same: members, finances, ministries, subdivisions, users, reports, and bilingual support.'
                                : 'Funcionalmente são idênticas. A Standalone não tem o directório público (que lista todas as denominações registadas na cloud), a página de registo, nem a landing page — porque nada disso faz sentido numa instalação privada. Todo o resto é igual: membros, finanças, ministérios, subdivisões, utilizadores, relatórios e suporte bilingue.' ?>
                            </p>
                        </div>
                    </div>
                    <div class="faq-item">
                        <button class="faq-q" onclick="toggleFaq(this)">
                            <?= $lang==='en'
                                ? 'What does the footer of the Standalone show?'
                                : 'O que mostra o rodapé da versão Standalone?' ?>
                            <span class="faq-arrow">▾</span>
                        </button>
                        <div class="faq-a">
                            <p><?= $lang==='en'
                                ? 'Every page of the Standalone shows at the bottom: the denomination name (e.g. "First Baptist Church © 2025") and below: "a product of sigedec.asaf.fyi" as a clickable link. This cannot be removed — it is part of the licence terms and serves as attribution to the original system.'
                                : 'Cada página da Standalone mostra no fundo: o nome da denominação (ex: "Igreja Baptista Central © 2025") e por baixo: "um produto da sigedec.asaf.fyi" como link clicável. Não pode ser removido — faz parte dos termos de licença e serve como atribuição ao sistema original.' ?>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

    </div>
</section>

<!-- ═══ FAQ ═════════════════════════════════════════════════ -->
<section class="land-section land-faq" id="faq">
    <div class="land-container">
        <div class="land-section-badge">❓ FAQ</div>
        <h2 class="land-h2"><?= $lang==='en' ? 'Frequently Asked Questions' : 'Perguntas Frequentes' ?></h2>
        <p class="land-section-sub"><?= $lang==='en'
            ? 'Everything you need to know before getting started. If your question is not here, write to us.'
            : 'Tudo o que precisa de saber antes de começar. Se a sua dúvida não estiver aqui, escreva-nos.' ?>
        </p>
        <div class="faq-grid">

          <!-- BLOCO: Registo e Acesso -->
          <div class="faq-block">
            <div class="faq-block-title"><span>🚀</span><?= $lang==='en' ? 'Registration & Access' : 'Registo e Acesso' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How do I register my denomination?' : 'Como registo a minha denominação?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Click "Register Denomination" on the homepage. Fill in the name of your denomination, country, province, district, and the administrator credentials. The password must have at least 6 characters including a letter, number and symbol (e.g. Igreja#1). You immediately get 6 months of free access — no credit card required.' : 'Clique em "Registar Denominação" na página inicial. Preencha o nome da denominação, país, província, distrito e as credenciais do administrador. A senha deve ter pelo menos 6 caracteres com letra, número e símbolo (ex: Igreja#1). Tem imediatamente 6 meses de acesso gratuito — sem cartão de crédito.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How does the login work?' : 'Como funciona o login?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Go to sigedec.asaf.fyi. The login form is at the top of the page. Enter your email and password. The system detects your role and redirects you to the correct dashboard: denomination admin, local manager, or finance manager.' : 'Aceda a sigedec.asaf.fyi. O formulário de login está no topo da página. Insira o email e a senha. O sistema detecta a sua função e redireciona-o para o painel correcto: admin da denominação, gestor local ou gestor financeiro.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Can multiple people access the same denomination?' : 'Podem várias pessoas aceder à mesma denominação?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Yes. The admin creates accounts for local managers (per congregation) and finance managers. Local managers only see their subdivision. Finance managers only see the finance module. The admin has full access.' : 'Sim. O admin cria contas para gestores locais (por congregação) e gestores financeiros. Os gestores locais apenas vêem a sua subdivisão. Os gestores financeiros apenas vêem o módulo financeiro. O admin tem acesso total.' ?></p></div></div>
          </div>

          <!-- BLOCO: Trial e Licença -->
          <div class="faq-block">
            <div class="faq-block-title"><span>🔑</span><?= $lang==='en' ? 'Trial & License' : 'Trial e Licença' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How long does the free trial last?' : 'Quanto tempo dura o período de teste gratuito?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? '6 months from registration. All features are available, same as the paid version. No obligations after the trial ends.' : '6 meses a partir do registo. Todas as funcionalidades estão disponíveis, igual à versão paga. Sem qualquer obrigação após o fim do trial.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'My trial is ending. How do I renew?' : 'O meu trial está a acabar. Como renovo?' ?><span class="faq-arrow">▾</span></button><div class="faq-a">
              <p><?= $lang==='en' ? 'Contact audivido3@gmail.com. After agreeing on payment you receive a license key (e.g. 1yr-XXXX-XXXX-XXXX-XXXX). Then:' : 'Contacte audivido3@gmail.com. Após o acordo de pagamento recebe uma chave de licença (ex: 1yr-XXXX-XXXX-XXXX-XXXX). Depois:' ?></p>
              <ol>
                <li><?= $lang==='en' ? 'Log in to your denomination panel' : 'Faça login no painel da sua denominação' ?></li>
                <li><?= $lang==='en' ? 'In the left menu click <strong>"🔑 License"</strong>' : 'No menu lateral clique em <strong>"🔑 Licença"</strong>' ?></li>
                <li><?= $lang==='en' ? 'Paste the key in the activation field' : 'Cole a chave no campo de activação' ?></li>
                <li><?= $lang==='en' ? 'Click <strong>"Activate"</strong> — access extended immediately' : 'Clique em <strong>"Activar"</strong> — acesso estendido de imediato' ?></li>
              </ol>
              <p><?= $lang==='en' ? 'Each key works only once. Used keys are rejected by the system.' : 'Cada chave funciona apenas uma vez. Chaves já usadas são rejeitadas pelo sistema.' ?></p>
            </div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'What happens when the trial expires?' : 'O que acontece quando o trial expira?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Access is suspended but your data is NOT deleted. Everything remains safe on the server. Once you activate a license key, all data is restored exactly as it was — members, finances, history. Nothing is lost.' : 'O acesso é suspenso mas os seus dados NÃO são eliminados. Tudo fica guardado no servidor. Quando activar uma chave de licença, tudo é restaurado exactamente como estava — membros, finanças, histórico. Não perde nada.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Can I get another trial period?' : 'Posso ter um novo período de teste?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Yes, in exceptional cases. Contact the administrator explaining your situation. If approved, a "trial" key is issued that extends access by another 6 months. Decided case by case.' : 'Sim, em casos excepcionais. Contacte o administrador explicando a situação. Se aprovado, é emitida uma chave "trial" que estende o acesso por mais 6 meses. Decidido caso a caso.' ?></p></div></div>
          </div>

          <!-- BLOCO: Gestão Interna -->
          <div class="faq-block">
            <div class="faq-block-title"><span>⚙️</span><?= $lang==='en' ? 'Internal Management' : 'Gestão Interna' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How do I register the church treasurer?' : 'Como registo o tesoureiro da igreja?' ?><span class="faq-arrow">▾</span></button><div class="faq-a">
              <p><?= $lang==='en' ? 'The denomination admin goes to <strong>"👥 Users"</strong> in the left menu → <strong>"+ New User"</strong>. Fill in name, email, password, and in "Role" select <strong>"💰 Finance Manager"</strong>. The treasurer logs in with their own email and sees only:' : 'O admin da denominação vai a <strong>"👥 Utilizadores"</strong> no menu lateral → <strong>"＋ Novo Utilizador"</strong>. Preenche nome, email, senha, e em "Função" selecciona <strong>"💰 Gestor Financeiro"</strong>. O tesoureiro entra com o seu próprio email e vê apenas:' ?></p>
              <ul>
                <li><?= $lang==='en' ? 'Income and expense categories' : 'Categorias de entrada e saída' ?></li>
                <li><?= $lang==='en' ? 'Record of all financial movements' : 'Registo de todos os movimentos financeiros' ?></li>
                <li><?= $lang==='en' ? 'Monthly summaries and balance' : 'Resumos mensais e saldo' ?></li>
                <li><?= $lang==='en' ? 'Printable financial reports' : 'Relatórios financeiros imprimíveis' ?></li>
              </ul>
              <p><?= $lang==='en' ? 'The treasurer cannot see members, subdivisions or any other section. Complete financial privacy.' : 'O tesoureiro não vê membros, subdivisões nem qualquer outra secção. Privacidade financeira total.' ?></p>
            </div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Can the local pastor only see his congregation?' : 'O pastor local pode ver apenas a sua congregação?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Exactly. Create a "Local Manager" account and assign the subdivision. From then on that user only sees, registers and manages members of that specific congregation — not others in the same denomination.' : 'Exactamente. Crie uma conta "Gestor Local" e atribua a subdivisão. A partir daí esse utilizador apenas vê, regista e gere os membros dessa congregação específica — não vê as outras da mesma denominação.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How do I change the system language?' : 'Como mudo o idioma do sistema?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'In the left menu of the dashboard, at the bottom, there is a "🌐 Português / English" button. Click it and the entire interface changes immediately. The change is visual only — all data remains intact. Each user can set their preferred language independently.' : 'No menu lateral do painel, no fundo, existe um botão "🌐 Português / English". Clique nele e toda a interface muda de imediato. A mudança é apenas visual — todos os dados ficam intactos. Cada utilizador define o idioma preferido de forma independente.' ?></p></div></div>
          </div>

          <!-- BLOCO: Dados e Privacidade -->
          <div class="faq-block">
            <div class="faq-block-title"><span>🔒</span><?= $lang==='en' ? 'Data & Privacy' : 'Dados e Privacidade' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Can another denomination see our data?' : 'Outra denominação pode ver os nossos dados?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Absolutely not. Each denomination is completely isolated. No other church — not even the system administrator — can access your members, finances or internal structure.' : 'De forma alguma. Cada denominação está completamente isolada. Nenhuma outra igreja — nem o administrador do sistema — consegue aceder aos seus membros, finanças ou estrutura interna.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Are passwords stored securely?' : 'As senhas são guardadas de forma segura?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Yes. All passwords are encrypted with bcrypt. Even with direct server access, only encrypted hashes are visible — never real passwords. Session cookies use HttpOnly and SameSite=Strict protection.' : 'Sim. Todas as senhas são encriptadas com bcrypt. Mesmo com acesso directo ao servidor, apenas hashes encriptados são visíveis — nunca as senhas reais. Os cookies de sessão usam protecção HttpOnly e SameSite=Strict.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'What happens to our data if we cancel?' : 'O que acontece aos nossos dados se cancelarmos?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Data remains on the server. To have it permanently deleted, send a deletion request to audivido3@gmail.com. The administrator removes all files within 48 hours and confirms by email.' : 'Os dados ficam no servidor. Para eliminação definitiva, envie um pedido para audivido3@gmail.com. O administrador remove todos os ficheiros em 48 horas e confirma por email.' ?></p></div></div>

          <!-- BLOCO: Portal do Membro -->
          <div class="faq-block">
            <div class="faq-block-title"><span>👤</span><?= $lang==='en' ? 'Member Portal' : 'Portal do Membro' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Can members access the system to view their own data?' : 'Os membros podem aceder ao sistema para ver os seus próprios dados?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Yes. Each member can have a dedicated portal login. The denomination admin or local pastor assigns a portal email and password to the member. When the member logs in at the main page using those credentials, they are taken directly to their personal dashboard — completely separate from the administrative area.' : 'Sim. Cada membro pode ter um acesso dedicado ao portal. O admin da denominação ou o pastor local atribui um email e senha de portal ao membro. Quando o membro entra na página principal com essas credenciais, é encaminhado directamente para o seu painel pessoal — completamente separado da área administrativa.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'What can the member see and do in their portal?' : 'O que é que o membro vê e pode fazer no seu portal?' ?><span class="faq-arrow">▾</span></button><div class="faq-a">
              <p><?= $lang==='en' ? 'The member can <strong>view</strong> (but not change):' : 'O membro pode <strong>consultar</strong> (mas não alterar):' ?></p>
              <ul>
                <li><?= $lang==='en' ? 'Full name, baptism date, birth date, marital status' : 'Nome completo, data de baptismo, data de nascimento, estado civil' ?></li>
                <li><?= $lang==='en' ? 'Membership status (Active, Suspended, Unlinked)' : 'Estado de membresia (Activo, Cortado, Desvinculado)' ?></li>
                <li><?= $lang==='en' ? 'Tithe/contribution classification (Regular, Irregular, Unfaithful)' : 'Classificação do dízimo/contribuições (Regular, Irregular, Infiel)' ?></li>
                <li><?= $lang==='en' ? 'Subdivision/congregation they belong to' : 'Subdivisão/congregação a que pertencem' ?></li>
                <li><?= $lang==='en' ? 'Suspension/cut history (if any)' : 'Histórico de cortes (se houver)' ?></li>
              </ul>
              <p><?= $lang==='en' ? 'The member can <strong>change</strong>:' : 'O membro pode <strong>alterar</strong>:' ?></p>
              <ul>
                <li><?= $lang==='en' ? 'Profile photo (max 2 MB — via camera or gallery)' : 'Foto de perfil (máx. 2 MB — pela câmara ou galeria)' ?></li>
                <li><?= $lang==='en' ? 'Own access password' : 'A sua própria senha de acesso' ?></li>
              </ul>
            </div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'What if the member has no photo when they first log in?' : 'O que acontece se o membro não tiver foto no primeiro acesso?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'The system detects the missing photo and shows a clear message inviting the member to add one. They can either take a photo in real time using their device camera, or choose an existing image from their phone gallery. The file is automatically compressed and the limit of 2 MB is enforced before saving.' : 'O sistema detecta a ausência de foto e exibe uma mensagem clara a convidar o membro a adicionar uma. O membro pode tirar uma foto na hora usando a câmara do dispositivo, ou escolher uma imagem existente da galeria do telemóvel. O ficheiro é processado automaticamente e o limite de 2 MB é verificado antes de gravar.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How do I set portal access for a member?' : 'Como atribuo acesso ao portal para um membro?' ?><span class="faq-arrow">▾</span></button><div class="faq-a">
              <p><?= $lang==='en' ? 'Go to <strong>Members</strong>, find the member, click <strong>Edit</strong>. In the editing modal, go to the <strong>"🔐 Portal"</strong> tab. Enter the email that the member will use to log in and a temporary password. Click Save. From that moment on, the member can log in using those credentials from the main SIGEDEC page.' : 'Vá a <strong>Membros</strong>, encontre o membro, clique em <strong>Editar</strong>. No modal de edição, vá ao separador <strong>"🔐 Portal"</strong>. Insira o email que o membro usará para entrar e uma senha temporária. Clique em Guardar. A partir desse momento, o membro pode entrar com essas credenciais pela página principal do SIGEDEC.' ?></p>
              <p><?= $lang==='en' ? 'To remove access, simply clear the portal email field and save.' : 'Para remover o acesso, basta limpar o campo de email do portal e guardar.' ?></p>
            </div></div>
          </div>

          <!-- BLOCO: Dízimo e Corte -->
          <div class="faq-block">
            <div class="faq-block-title"><span>🙏</span><?= $lang==='en' ? 'Tithe & Discipline' : 'Dízimo e Disciplina' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How does tithe/contribution status classification work?' : 'Como funciona a classificação da situação do dízimo/contribuições?' ?><span class="faq-arrow">▾</span></button><div class="faq-a">
              <p><?= $lang==='en' ? 'The finance manager goes to <strong>Finance → Tithe Status</strong>. The screen shows all active members. For each member, the manager selects one of three classifications:' : 'O gestor financeiro vai a <strong>Financeiro → Situação Dízimo</strong>. O ecrã mostra todos os membros activos. Para cada membro, o gestor selecciona uma de três classificações:' ?></p>
              <ul>
                <li><strong><?= $lang==='en' ? 'Regular' : 'Regular' ?></strong> — <?= $lang==='en' ? 'contributes consistently and on time' : 'contribui de forma consistente e pontual' ?></li>
                <li><strong><?= $lang==='en' ? 'Irregular' : 'Irregular' ?></strong> — <?= $lang==='en' ? 'contributions oscillate (sometimes contributes, sometimes not)' : 'as contribuições oscilam (às vezes contribui, às vezes não)' ?></li>
                <li><strong><?= $lang==='en' ? 'Unfaithful' : 'Infiel' ?></strong> — <?= $lang==='en' ? 'stopped contributing entirely for more than 3 consecutive months' : 'parou de contribuir totalmente há mais de 3 meses consecutivos' ?></li>
              </ul>
              <p><?= $lang==='en' ? 'This classification is visible to the member in their portal. It helps the pastoral leadership have an honest picture without exposing exact financial amounts to third parties.' : 'Esta classificação é visível ao membro no seu portal. Ajuda a liderança pastoral a ter um quadro honesto sem expor valores financeiros exactos a terceiros.' ?></p>
            </div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Who can suspend a member from communion or unlink them?' : 'Quem pode cortar um membro da comunhão ou desvinculá-lo?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'This is a pastoral authority action. In SIGEDEC, the <strong>Local Manager</strong> — who represents the pastor or local leader of a congregation — has permission to change a member\'s status to "Suspended" or "Unlinked". The denomination admin also has this permission. The finance manager and the member themselves cannot perform this action. This mirrors the real authority structure in most African denominations: the local pastor makes discipline decisions for their congregation.' : 'Esta é uma acção de autoridade pastoral. No SIGEDEC, o <strong>Gestor Local</strong> — que representa o pastor ou líder local de uma congregação — tem permissão para alterar o estado de um membro para "Cortado" ou "Desvinculado". O admin da denominação também tem essa permissão. O gestor financeiro e o próprio membro não podem realizar esta acção. Isto espelha a estrutura real de autoridade na maioria das denominações africanas: o pastor local toma decisões de disciplina para a sua congregação.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'What is the difference between "Suspended" and "Unlinked"?' : 'Qual é a diferença entre "Cortado" e "Desvinculado"?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? '"Suspended" means the member is under a temporary disciplinary measure — communion is withdrawn but they remain in the denomination\'s records. It is reversible: when the discipline period ends, the pastor restores them to Active. "Unlinked" means the member has formally left or been formally separated from the denomination. The record remains in the system for historical integrity, but they are no longer counted as an active member.' : '"Cortado" significa que o membro está sob uma medida disciplinar temporária — a comunhão é retirada mas permanece nos registos da denominação. É reversível: quando o período de disciplina termina, o pastor restaura-o para Activo. "Desvinculado" significa que o membro saiu formalmente ou foi separado formalmente da denominação. O registo permanece no sistema por integridade histórica, mas já não é contabilizado como membro activo.' ?></p></div></div>
          </div>

          <!-- BLOCO: Backup -->
          <div class="faq-block">
            <div class="faq-block-title"><span>💾</span><?= $lang==='en' ? 'Backup & Data Export' : 'Backup e Exportação de Dados' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How do I back up my denomination\'s data?' : 'Como faço o backup dos dados da minha denominação?' ?><span class="faq-arrow">▾</span></button><div class="faq-a">
              <p><?= $lang==='en' ? 'Log in as denomination administrator. In the left menu, click <strong>"💾 Backup"</strong>. A page shows the date of your last backup. Click <strong>"⬇️ Download Backup"</strong>. The system generates a ZIP file containing:' : 'Entre como administrador da denominação. No menu lateral, clique em <strong>"💾 Backup"</strong>. Uma página mostra a data do último backup efectuado. Clique em <strong>"⬇️ Descarregar Backup"</strong>. O sistema gera um ficheiro ZIP contendo:' ?></p>
              <ul>
                <li><?= $lang==='en' ? 'All member records (name, baptism, status, history...)' : 'Todos os registos de membros (nome, baptismo, estado, histórico...)' ?></li>
                <li><?= $lang==='en' ? 'Member profile photos' : 'Fotos de perfil dos membros' ?></li>
                <li><?= $lang==='en' ? 'Financial transactions and categories' : 'Transacções financeiras e categorias' ?></li>
                <li><?= $lang==='en' ? 'Subdivision structure and leaders' : 'Estrutura de subdivisões e líderes' ?></li>
                <li><?= $lang==='en' ? 'Ministries and assignments' : 'Ministérios e atribuições' ?></li>
              </ul>
              <p><?= $lang==='en' ? 'Save this ZIP in a safe place outside the system — a USB drive, Google Drive, or another cloud storage.' : 'Guarde este ZIP num lugar seguro fora do sistema — uma pen USB, Google Drive, ou outro armazenamento cloud.' ?></p>
            </div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How often should I do a backup?' : 'Com que frequência devo fazer backup?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'We recommend at least once a month, or after any significant update — registering many new members, a major financial entry, or an important discipline action. The backup page always shows the date of the last download so you know if it is up to date.' : 'Recomendamos pelo menos uma vez por mês, ou após qualquer actualização significativa — registo de muitos novos membros, um lançamento financeiro importante, ou uma acção disciplinar relevante. A página de backup mostra sempre a data da última descarga para saber se está actualizado.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'What can the system administrator back up?' : 'O que é que o administrador do sistema pode fazer em termos de backup?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'The general system administrator (super admin) has access to a <strong>Full System Backup</strong> that packages the data of all registered denominations into a single ZIP file. This serves as a global archive for disaster recovery. Each denomination\'s data remains isolated within the ZIP — there is no mixing. This feature is only available to the super admin, not to individual denomination administrators.' : 'O administrador geral do sistema (super admin) tem acesso a um <strong>Backup Geral do Sistema</strong> que empacota os dados de todas as denominações registadas num único ficheiro ZIP. Serve como arquivo global para recuperação em caso de desastre. Os dados de cada denominação ficam isolados dentro do ZIP — não há mistura. Esta funcionalidade é exclusiva do super admin, não está disponível aos admins de denominação individuais.' ?></p></div></div>
          </div>

          </div>

          <!-- BLOCO: Funcionalidades -->
          <div class="faq-block">
            <div class="faq-block-title"><span>📋</span><?= $lang==='en' ? 'Features' : 'Funcionalidades' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How does member suspension and restoration work?' : 'Como funciona o corte e restauração de um membro?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'When suspending a member, the system requires the reason and circumstances. Each suspension is logged with date and counter (1st, 2nd, 3rd...). If expelled definitively, the member goes to the private blacklist — never deleted. When they return, click "Restore" to reinstate active status.' : 'Ao cortar um membro, o sistema exige o motivo e as circunstâncias. Cada corte fica registado com data e contador (1.º, 2.º, 3.º...). Se expulso definitivamente, vai para a lista negra privada — nunca é eliminado. Quando voltar, clique "Restaurar" para devolver o estado activo.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How does courtship and marriage tracking work?' : 'Como funciona o acompanhamento de namoro e casamento?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Update the member status to "Dating", enter the partner name and the public announcement date. On the wedding day, change both to "Married" and link them as spouses. The system updates both records automatically.' : 'Actualiza o estado do membro para "Em Namoro", insere o nome do parceiro e a data de apresentação pública. No dia do casamento, altera ambos para "Casado" e associa-os como cônjuges. O sistema actualiza ambas as fichas automaticamente.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Can I print reports and financial statements?' : 'Posso imprimir relatórios e extractos financeiros?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Yes. All reports have a "🖨️ Print / PDF" button. Click it, your browser opens the print dialog. Choose "Save as PDF" to generate a file or send to a printer. Works on both mobile and desktop.' : 'Sim. Todos os relatórios têm o botão "🖨️ Imprimir / PDF". Ao clicar, o browser abre o diálogo de impressão. Escolha "Guardar como PDF" ou envie para uma impressora. Funciona em telemóvel e computador.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Does it need an app? Does it work without internet?' : 'Precisa de app? Funciona sem internet?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'No installation required — runs entirely in the browser (Chrome, Firefox, Safari, Edge). Needs an internet connection since data is stored on the server. Works perfectly on any modern smartphone.' : 'Não precisa de instalação — corre no browser (Chrome, Firefox, Safari, Edge). Precisa de internet pois os dados ficam no servidor. Funciona perfeitamente em qualquer telemóvel moderno.' ?></p></div></div>
          </div>

          <!-- BLOCO: Pagamento -->
          <div class="faq-block">
            <div class="faq-block-title"><span>💰</span><?= $lang==='en' ? 'Payment' : 'Pagamento' ?></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'How do I pay for the license?' : 'Como faço o pagamento da licença?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'Contact audivido3@gmail.com. Payment method is flexible and agreed case by case: M-Pesa, Airtel Money, Orange Money, bank transfer, Western Union, or another method available in your country. Prices in USD simplify cross-border transactions.' : 'Contacte audivido3@gmail.com. O método de pagamento é flexível e combinado caso a caso: M-Pesa, Airtel Money, Orange Money, transferência bancária, Western Union, ou outro método disponível no seu país. Os preços em dólares simplificam as transacções transfronteiriças.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Are there discounts for small churches?' : 'Há descontos para igrejas com poucos recursos?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'The prices ($10/$15/$20) are already symbolic. For small rural congregations or missions in economically challenged regions, contact us — we will always try to find a solution. Our mission is to serve the African church, not make it inaccessible.' : 'Os preços ($10/$15/$20) já são simbólicos. Para pequenas congregações rurais ou missões em regiões economicamente difíceis, contacte-nos — tentaremos sempre encontrar uma solução. A nossa missão é servir a igreja africana, não torná-la inacessível.' ?></p></div></div>
            <div class="faq-item"><button class="faq-q" onclick="toggleFaq(this)"><?= $lang==='en' ? 'Why charge at all if it is for the church?' : 'Porquê cobrar se é para a igreja?' ?><span class="faq-arrow">▾</span></button><div class="faq-a"><p><?= $lang==='en' ? 'The system has monthly server costs: hosting, domain, security, maintenance. The $10/year (less than $1/month) covers only these costs. There is no profit. The goal is sustainability so the system can continue serving African churches for years to come.' : 'O sistema tem custos mensais de servidor: hosting, domínio, segurança, manutenção. Os $10/ano (menos de $1/mês) cobrem apenas esses custos. Não há lucro. O objectivo é a sustentabilidade para que o sistema continue a servir as igrejas africanas por muitos anos.' ?></p></div></div>
          </div>

        </div><!-- /faq-grid -->
    </div>
</section>

<section class="land-section land-contact" id="contacto">
    <div class="land-container">
        <div class="land-section-badge">📬 <?= $lang==='en' ? 'Contact' : 'Contacto' ?></div>
        <h2 class="land-h2"><?= $lang==='en' ? 'Questions? We are here.' : 'Dúvidas? Estamos aqui.' ?></h2>
        <p class="land-section-sub"><?= $lang==='en'
            ? 'To acquire a license, extend a trial, report an issue or just ask a question — contact us directly.'
            : 'Para adquirir uma licença, estender um período de teste, reportar um problema ou apenas fazer uma pergunta — contacte-nos directamente.' ?>
        </p>
        <div class="land-contact-grid">
            <div class="land-contact-item">
                <div class="land-contact-icon">📧</div>
                <h3>Email</h3>
                <a href="mailto:<?= CONTACT_EMAIL ?>"><?= CONTACT_EMAIL ?></a>
            </div>
            <div class="land-contact-item">
                <div class="land-contact-icon">🌐</div>
                <h3>Website</h3>
                <a href="<?= SITE_URL ?>"><?= SITE_URL ?></a>
            </div>
            <div class="land-contact-item">
                <div class="land-contact-icon">📍</div>
                <h3><?= $lang==='en' ? 'Based in' : 'Com base em' ?></h3>
                <span>Africa 🌍</span>
            </div>
        </div>
    </div>
</section>

<!-- ═══ RODAPÉ ════════════════════════════════════════════════════ -->
<!-- ═══ BANNER SUGESTÕES ══════════════════════════════════════ -->

<!-- ═══ CONTEÚDO EDITORIAL RICO (SEO + AdSense) ══════════════════ -->
<section class="land-editorial" id="sobre-sigedec" aria-label="Sobre o SIGEDEC">
<div class="land-container land-editorial-inner">

<article class="land-article">
<h2>O SIGEDEC: Um Sistema de Gestão Completo para Igrejas Africanas</h2>
<p>O SIGEDEC nasceu de uma necessidade real: as igrejas e denominações religiosas em África precisavam de uma solução digital acessível, disponível nos idiomas locais, e adaptada às estruturas organizacionais únicas das comunidades de fé africanas. Ao contrário das soluções ocidentais que não reconhecem subdivisões como congregações locais, grupos de zona ou associações regionais, o SIGEDEC foi construído com a realidade africana como ponto de partida.</p>

<p>Hoje, o sistema oferece uma plataforma completa para gerir todos os aspectos administrativos de uma denominação religiosa. Desde o registo detalhado de cada membro — com fotografia, estado civil, data de baptismo, ministério, consagrações e histórico de cortes e restaurações — até ao módulo financeiro que regista dízimos, ofertas, doações e despesas com relatórios mensais imprimíveis. Tudo numa interface disponível em Português e em Inglês, adaptada para funcionar perfectamente em telemóvel sem necessidade de instalar qualquer aplicação.</p>

<h3>Gestão de Membros com Profundidade Real</h3>
<p>A ficha de cada membro no SIGEDEC vai muito além de um nome e contacto. O sistema regista o estado espiritual (activo, em prova, cortado, desvinculado), o estado civil com acompanhamento de namorados, noivos e casamentos anunciados publicamente na igreja, os ministérios onde o membro serve, as consagrações recebidas (diácono, presbítero, evangelista, pastor), e o histórico completo de movimentos. Quando um membro é cortado, o sistema exige o registo do motivo e das circunstâncias, criando um histórico auditável que serve a liderança da denominação.</p>

<p>A lista negra é privada e apenas visível ao administrador da denominação. Quando um membro é restaurado — seja depois de um período de prova, seja depois de uma reconciliação formal — a restauração também fica registada com data e testemunhos. Esta rastreabilidade é essencial para denominações com múltiplas congregações que precisam de saber o historial de um membro antes de o receber.</p>

<h3>Privacidade e Segurança de Dados</h3>
<p>Cada denominação registada no SIGEDEC existe num espaço completamente isolado. Os dados de uma denominação nunca são visíveis a outra — nem ao próprio administrador geral do sistema. As senhas de todos os utilizadores são encriptadas com o algoritmo bcrypt, um padrão de segurança da indústria. Os cookies de sessão usam protecção HttpOnly e SameSite=Strict para impedir ataques de sequestro de sessão.</p>

<p>O módulo financeiro tem uma camada adicional de privacidade: o tesoureiro tem acesso exclusivo às finanças e não vê a lista de membros, e os gestores locais vêem apenas a sua congregação, nunca as finanças. Esta separação de responsabilidades é fundamental para manter a confiança dentro de uma organização religiosa.</p>

<h3>Subdivisões e Hierarquia Denominacional</h3>
<p>Uma das funcionalidades mais valorizadas pelas denominações com múltiplas congregações é o sistema de subdivisões. O administrador da denominação cria as subdivisões (congregações locais, grupos de zona, distritos ou regiões) e atribui um gestor local a cada uma. Cada gestor local acede ao sistema com as suas próprias credenciais e vê apenas os membros da sua congregação. O administrador central tem visão completa de toda a denominação, com estatísticas globais e relatórios consolidados.</p>

<p>Este modelo reflecte a estrutura real de denominações como as Assembleias de Deus, a Igreja Universal, as Igrejas Baptistas, as denominações Pentecostais e Carismáticas, e muitas outras presentes em Moçambique, Angola, Congo, Tanzânia, Uganda, Quénia, África do Sul e nos 54 países do continente africano.</p>
</article>

<article class="land-article">
<h2>SIGEDEC Standalone: Instale o Sistema no Servidor da Sua Denominação</h2>
<p>Para denominações que preferem total autonomia sobre os seus dados, ou que já possuem um servidor web ou serviço de alojamento próprio, o SIGEDEC oferece a Edição Autónoma (Standalone). Trata-se de um pacote completo do sistema — idêntico em funcionalidades à versão cloud — que pode ser descarregado, instalado e executado no domínio da denominação, sem dependência do servidor central da plataforma SIGEDEC.</p>

<p>A analogia mais próxima é o WordPress: tal como se descarrega o WordPress e se instala num servidor pessoal para criar um site próprio, o SIGEDEC Standalone permite à denominação ter o seu sistema de gestão num endereço como <em>gestao.suaigreja.com</em> ou <em>suaigreja.com/gestao</em>, com total controlo sobre os dados e sobre quem tem acesso.</p>

<h3>Como Funciona a Activação</h3>
<p>O SIGEDEC Standalone utiliza um sistema de chave de activação para proteger a propriedade intelectual do software. Após adquirir a licença, a denominação recebe uma chave única no formato <em>standalone-XXXX-XXXX-XXXX-XXXX</em>. Esta chave é inserida na primeira vez que o sistema é acedido e é validada em tempo real contra o servidor central da SIGEDEC. No momento da validação, a chave fica permanentemente associada ao domínio onde foi activada — tornando impossível a sua utilização noutro domínio sem autorização expressa.</p>

<p>O sistema verifica silenciosamente a licença a cada sete dias. Se o servidor de verificação não for contactável (por exemplo, por falha temporária de rede), o sistema aceita até dez dias de funcionamento sem verificação, garantindo continuidade mesmo em contextos com acesso intermitente à internet — uma realidade frequente em muitas regiões de África.</p>

<h3>Preço e Forma de Pagamento</h3>
<p>A licença da Edição Autónoma custa vinte dólares americanos ($20 USD), num pagamento único e definitivo. Não existem mensalidades, taxas de renovação ou custos ocultos. O sistema fica disponível para uso permanente na denominação após o pagamento. Para facilitar o acesso a igrejas em diferentes países africanos, o pagamento pode ser efectuado através dos métodos mais populares no continente: M-Pesa, Airtel Money, Orange Money, transferência bancária local ou internacional, e Western Union. Os detalhes são combinados directamente com o administrador da plataforma através do endereço de email audivido3@gmail.com.</p>

<h3>O Rodapé da Edição Autónoma</h3>
<p>Em todas as páginas do sistema Standalone, o rodapé exibe o nome da denominação que adquiriu a licença — por exemplo, "Igreja Baptista Central © 2025" — seguido da linha "um produto da sigedec.asaf.fyi" com um link clicável para o site do sistema original. Este elemento de atribuição faz parte dos termos de licença e não pode ser removido. Serve simultaneamente como crédito ao desenvolvimento original e como forma de outras denominações descobrirem a plataforma.</p>

<h3>Requisitos Técnicos Mínimos</h3>
<p>O SIGEDEC Standalone foi desenhado para funcionar em praticamente qualquer serviço de alojamento web disponível em África. Os requisitos são: PHP na versão 8.0 ou superior (incluído na esmagadora maioria dos planos de hosting partilhado actuais), um servidor web como Apache, Nginx ou LiteSpeed (também incluído por defeito nos hostings partilhados), e ligação à internet para a verificação periódica da licença. Não é necessária nenhuma base de dados MySQL — o sistema usa ficheiros JSON para armazenar todos os dados, o que simplifica radicalmente a instalação e a manutenção.</p>
</article>

<article class="land-article">
<h2>Planos e Preços da Versão Cloud</h2>
<p>A versão cloud do SIGEDEC, alojada em sigedec.asaf.fyi, oferece quatro opções para as denominações que pretendem usar o sistema sem se preocupar com instalação ou manutenção de servidor.</p>

<p>O <strong>período de teste gratuito</strong> dura seis meses completos e inclui todas as funcionalidades sem qualquer restrição. Não é necessário cartão de crédito para iniciar o teste — basta registar a denominação com um email válido e criar uma senha segura. O sistema está disponível de imediato após o registo.</p>

<p>O <strong>plano anual</strong> custa dez dólares por ano ($10/ano), o que equivale a menos de um dólar por mês. Este plano inclui acesso ilimitado a todas as funcionalidades, sem limite de membros, sem limite de utilizadores e sem limite de transacções financeiras. É activado através de uma chave de licença enviada por email após confirmação do pagamento.</p>

<p>O <strong>plano bienal</strong> custa quinze dólares por dois anos ($15 por 24 meses), representando uma poupança de cinco dólares face a dois planos anuais consecutivos. O <strong>plano trienal</strong>, o mais económico em termos proporcionais, custa vinte dólares por três anos ($20 por 36 meses), o que equivale a aproximadamente cinquenta e seis cêntimos por mês — um valor verdadeiramente simbólico para uma ferramenta de gestão completa.</p>

<h3>O SIGEDEC e o AdSense do Google</h3>
<p>O site sigedec.asaf.fyi está a candidatar-se ao programa Google AdSense como forma de ajudar a suportar os custos de desenvolvimento e manutenção da plataforma. O Google AdSense é um programa de publicidade contextual que exibe anúncios relevantes para os visitantes do site, gerando uma pequena receita que é reinvestida na melhoria do sistema. Esta receita é complementar às licenças de uso e não afecta os preços praticados para as denominações. O SIGEDEC compromete-se a nunca exibir publicidade dentro das áreas autenticadas do sistema — os anúncios aparecem apenas nas páginas públicas do site de apresentação.</p>

<h3>Sobre a Privacidade e os Cookies</h3>
<p>O SIGEDEC utiliza apenas cookies estritamente necessários para o funcionamento do sistema: o cookie de sessão (que mantém o utilizador autenticado enquanto navega) e uma preferência de idioma (que recorda se o utilizador prefere Português ou Inglês). Não são utilizados cookies de rastreamento, cookies de publicidade de terceiros ou qualquer outro mecanismo de rastreamento de comportamento. A política de privacidade completa está disponível em sigedec.asaf.fyi/privacy.php e a política de cookies em sigedec.asaf.fyi/cookies.php.</p>
</article>

<!-- ─── DOWNLOAD STANDALONE ─── -->
<article class="land-article land-article-download" id="download-standalone">
<h2>⬇️ Descarregar o SIGEDEC Standalone Edition</h2>
<p>Se a sua denominação pretende instalar o sistema de gestão SIGEDEC no seu próprio servidor ou domínio, o processo começa por solicitar a licença ao administrador da plataforma. O download do ficheiro ZIP é gratuito e imediato — clique no botão acima para descarregar. O sistema, porém, só fica operacional após inserir uma chave de activação válida, que é adquirida separadamente pelo preço único de $20 USD. Esta abordagem é idêntica à do WordPress: descarrega-se livremente, mas para publicar num domínio real é necessário um serviço de alojamento. No SIGEDEC, o "alojamento" é a chave de activação que valida a instalação no seu domínio específico.</p>

<div class="editorial-download-box">
    <div class="editorial-download-info">
        <div class="editorial-download-label">📦 SIGEDEC Standalone Edition</div>
        <div class="editorial-download-meta">PHP 8.0+ &middot; Qualquer hosting partilhado &middot; ~70KB &middot; Sem base de dados MySQL</div>
        <div class="editorial-download-price">$20 USD &mdash; pagamento único, sem mensalidades, sem data de validade</div>
    </div>
    <div style="display:flex;flex-direction:column;gap:.65rem;align-items:center">
        <a href="/downloads/download.php" class="btn-editorial-download" download>
            ⬇️ Descarregar Grátis
        </a>
        <a href="mailto:audivido3@gmail.com?subject=SIGEDEC%20Standalone%20%E2%80%94%20Chave%20de%20Activa%C3%A7%C3%A3o" style="font-size:.8rem;color:#c9a84c;text-align:center">
            🔑 Obter chave de activação — $20 →
        </a>
    </div>
</div>

<p>Após enviar o email a solicitar a licença Standalone, o administrador confirma a disponibilidade e indica os métodos de pagamento aceites (M-Pesa, Airtel Money, Orange Money, transferência bancária ou Western Union). Depois do pagamento confirmado, recebe por email o link de download do ficheiro ZIP e a chave de activação única no prazo máximo de 24 horas.</p>

<h3>O Que Está Incluído no Ficheiro ZIP</h3>
<p>O ficheiro ZIP do SIGEDEC Standalone contém a estrutura completa do sistema pronta a usar. Dentro encontra as pastas <em>church</em> (gestão de membros, líderes, ministérios, subdivisões, utilizadores e relatórios), <em>finance</em> (categorias financeiras, registo de movimentos e relatórios financeiros), <em>local</em> (painel do gestor local e lista de membros por congregação), <em>includes</em> (ficheiros de configuração, lógica de dados, layout e idiomas), <em>assets</em> (CSS, JavaScript e fontes) e <em>data</em> (pasta onde os dados da denominação são guardados em formato JSON, protegida por .htaccess).</p>

<p>Também está incluído um ficheiro <em>README.md</em> com instruções detalhadas de instalação passo a passo, que qualquer pessoa com conhecimentos básicos de cPanel ou de upload por FTP consegue seguir sem dificuldade. O ficheiro <em>activate.php</em> é a primeira página que aparece após o upload — é aí que se insere a chave de activação recebida por email. O <em>install.php</em> é o assistente de configuração inicial que pede o nome da denominação e as credenciais do administrador. O <em>index.php</em> da raiz é a página de login que os utilizadores verão em todas as visitas subsequentes.</p>

<h3>Passo a Passo de Instalação</h3>
<p><strong>Primeiro passo — Upload dos ficheiros:</strong> Aceda ao painel de controlo do seu hosting (tipicamente o cPanel, DirectAdmin ou Plesk). Abra o Gestor de Ficheiros ou ligue-se via FTP com um cliente como o FileZilla. Navegue até à pasta pública do seu domínio (geralmente chamada <em>public_html</em>, <em>www</em> ou <em>htdocs</em>). Faça upload do ficheiro ZIP e extraia-o directamente no servidor, ou extraia primeiro no computador e depois faça upload dos ficheiros todos. Se pretende instalar numa subpasta (por exemplo em <em>gestao.suaigreja.com/sistema</em>), crie primeiro essa pasta e faça o upload para dentro dela.</p>

<p><strong>Segundo passo — Abrir o browser:</strong> Após o upload, abra o browser e aceda ao URL onde instalou o sistema. Se instalou na raiz do domínio, aceda a <em>suaigreja.com</em>. Se instalou numa subpasta, aceda a <em>suaigreja.com/gestao</em>. O sistema detecta automaticamente que ainda não está activado e redireciona para a página de activação em <em>/activate.php</em>.</p>

<p><strong>Terceiro passo — Activação:</strong> Na página de activação verá um formulário com um campo de texto onde deve colar a chave de activação recebida por email. A página também mostra o domínio detectado — confirme que corresponde ao domínio correcto antes de continuar. Clique em "Activar". O sistema envia um pedido ao servidor central da SIGEDEC em <em>sigedec.asaf.fyi</em>, que valida a chave e verifica se ainda não foi utilizada noutro domínio. Se tudo estiver correcto, a chave é activada e associada permanentemente ao seu domínio. Este processo demora normalmente menos de cinco segundos.</p>

<p><strong>Quarto passo — Configuração inicial:</strong> Após a activação, o sistema redireciona para o assistente de instalação em <em>/install.php</em>. Aqui preenche o nome completo da sua denominação (por exemplo "Igreja Baptista da Graça — Sede"), o país, e as credenciais da conta de administrador: nome, email e senha. A senha deve ter pelo menos seis caracteres e incluir pelo menos uma letra, um número e um símbolo especial (por exemplo Igreja#2025). Clique em "Concluir Instalação". O sistema cria automaticamente toda a estrutura de dados necessária e redireciona para a página de login.</p>

<p><strong>Quinto passo — Primeiro login e personalização:</strong> Entre com o email e senha definidos no passo anterior. Será encaminhado para o painel de administração central. Antes de começar a registar membros, é recomendável configurar as subdivisões da denominação (congregações locais, zonas ou distritos) em <em>Configurações → Subdivisões</em>, as categorias financeiras em <em>Finanças → Categorias</em> (por exemplo: Dízimos, Oferta do Dia do Senhor, Fundo de Construção, Despesas Administrativas), e criar as contas dos outros utilizadores do sistema (gestores locais e tesoureiro) em <em>Utilizadores</em>.</p>
</article>

<article class="land-article">
<h2>Funcionalidades Detalhadas do Sistema SIGEDEC</h2>

<h3>Registo e Acompanhamento de Membros</h3>
<p>O módulo de membros é o núcleo do sistema. Cada ficha de membro inclui: nome completo, número de identificação, data de nascimento, género, estado civil, número de telefone, endereço, fotografia, data de conversão, data de baptismo em água, data de baptismo no Espírito Santo, subdivisão a que pertence, ministério(s) onde serve, consagrações recebidas com data de cada uma, estado actual (activo, em prova, cortado, desvinculado), e histórico completo de alterações de estado com data, motivo e circunstâncias registadas pelo administrador.</p>

<p>O sistema acompanha também o estado relacional dos membros solteiros. Quando um casal é apresentado publicamente na congregação, o administrador actualiza o estado de ambos para "Em Namoro" e regista a data da apresentação. Quando se casam, o estado de ambos é actualizado para "Casado" e os dois perfis ficam ligados entre si. Esta funcionalidade é especialmente valorizada em denominações onde o processo de namoro e casamento tem um acompanhamento pastoral formal.</p>

<h3>Lista Negra e Gestão de Expulsões</h3>
<p>Quando um membro é formalmente expulso da denominação, o administrador registra a decisão com o motivo completo, a data da decisão disciplinar, e eventuais testemunhas ou decisão de conselho. O membro passa para a lista negra da denominação — um registo privado, visível apenas ao administrador central, que contém todos os membros com processos disciplinares em curso ou concluídos com expulsão definitiva.</p>

<p>A lista negra serve como salvaguarda: quando outra congregação da mesma denominação recebe um pedido de transferência de membro, o administrador pode verificar o histórico antes de aceitar a recepção. Se um membro expulso se arrepender e procurar restauração, o administrador pode iniciar um processo de restauração no próprio sistema, registando as condições, o período de prova e a data de restauração efectiva ao estado activo.</p>

<h3>Relatórios e Estatísticas</h3>
<p>O módulo de relatórios gera documentos detalhados para diferentes necessidades: lista completa de membros activos com todos os dados de contacto, lista de membros por subdivisão, relatório de novos membros no período seleccionado, relatório de membros cortados e restaurados, estatísticas por faixa etária e por género, lista de membros em namoro e casais recentes, relatório de líderes e ministérios com os membros afectos a cada um, e resumo geral da denominação com totais globais.</p>

<p>Todos os relatórios têm um botão de impressão que abre o diálogo nativo do browser. O utilizador pode imprimir directamente numa impressora ou guardar como ficheiro PDF usando a opção "Guardar como PDF" do browser — disponível em Chrome, Firefox, Safari e Edge, tanto em computador como em telemóvel. O layout de impressão foi desenhado especificamente para papel A4 e elimina automaticamente a navegação, os botões e todos os elementos de interface que não fazem sentido num documento impresso.</p>

<h3>Módulo Financeiro Completo</h3>
<p>O módulo financeiro do SIGEDEC permite à denominação ter um registo rigoroso e transparente de todas as entradas e saídas. O tesoureiro — ou o administrador, se preferir gerir as finanças directamente — regista cada transacção com: data, tipo (entrada ou saída), valor em moeda local, categoria (previamente configurada pelo administrador), descrição detalhada e notas adicionais se necessário.</p>

<p>O sistema calcula automaticamente o saldo mensal e acumulado. A vista mensal mostra todas as transacções do mês seleccionado, o total de entradas, o total de saídas e o saldo resultante. Os relatórios financeiros são filtráveis por mês e por categoria, permitindo por exemplo ver apenas os dízimos do primeiro trimestre, ou todas as despesas de manutenção do ano. O relatório financeiro imprimível inclui o cabeçalho com o nome da denominação, o período, os totais e a listagem detalhada de movimentos.</p>

<h3>Múltiplos Utilizadores com Diferentes Permissões</h3>
<p>O sistema suporta três níveis de acesso distintos, cada um com uma vista completamente diferente do sistema. O <strong>administrador da denominação</strong> tem acesso total: vê todos os membros de todas as subdivisões, todas as finanças, todos os relatórios, pode criar e eliminar utilizadores, configurar subdivisões e ministérios, e gerir a lista negra. O <strong>gestor local</strong> (pastor de congregação ou responsável de zona) vê apenas os membros da sua subdivisão específica, pode registar novos membros, actualizar estados e gerar relatórios da sua congregação — nunca vê dados de outras congregações nem das finanças. O <strong>gestor financeiro</strong> (tesoureiro) vê exclusivamente o módulo financeiro: registar entradas e saídas, gerir categorias e gerar relatórios financeiros — não tem acesso à lista de membros, subdivisões ou qualquer outra secção da gestão eclesiástica.</p>

<p>Esta separação de responsabilidades não é apenas uma questão técnica — reflecte a realidade organizacional de uma denominação religiosa, onde a informação sobre membros individuais tem um carácter pastoral e confidencial, e a informação financeira exige transparência com a liderança mas privacidade face ao corpo geral de membros.</p>

<h3>Interface Bilingue Português — Inglês</h3>
<p>O botão de mudança de idioma está sempre visível no menu lateral do painel de controlo. Ao clicar, toda a interface muda imediatamente para o outro idioma — todos os menus, botões, etiquetas, mensagens de erro, mensagens de sucesso, relatórios e assistentes. A mudança é instantânea e não requer recarregar a página. A preferência de idioma é guardada no browser do utilizador, pelo que na próxima visita o sistema aparece automaticamente no idioma preferido.</p>

<p>Esta funcionalidade bilingue é especialmente relevante em países como Moçambique e Angola, onde o Português é língua oficial mas muitos membros e líderes provenientes de países anglófonos vizinhos (Zimbabué, Malawi, Tanzânia, Zâmbia) preferem trabalhar em Inglês. Numa mesma denominação com congregações em diferentes países, cada utilizador pode trabalhar no idioma que domina melhor.</p>
</article>

<article class="land-article">
<h2>Perguntas Sobre a Edição Standalone — Guia Completo</h2>

<h3>Preciso de conhecimentos técnicos para instalar?</h3>
<p>Não. O processo de instalação foi desenhado para ser realizável por qualquer pessoa que saiba usar um painel de controlo de hosting (cPanel, DirectAdmin) ou um cliente FTP. O README incluído no ZIP explica cada passo com clareza. Se nunca fez upload de ficheiros para um servidor, o processo pode demorar entre 15 e 30 minutos na primeira vez. Não é necessário saber programar, configurar bases de dados, editar ficheiros de configuração ou usar a linha de comandos.</p>

<h3>E se o meu hosting não suportar PHP 8.0?</h3>
<p>A maioria dos hostings actuais já inclui PHP 8.0 ou superior por defeito. Se o seu hosting usar uma versão mais antiga, pode mudar a versão do PHP no cPanel em menos de um minuto sem afectar outros sites no mesmo servidor. Se tiver dúvidas, contacte o suporte do seu hosting e diga que precisa de PHP 8.0 para executar uma aplicação web — é um pedido completamente normal que qualquer hosting consegue satisfazer.</p>

<h3>Os meus dados ficam a salvo se o servidor da SIGEDEC fechar?</h3>
<p>Sim. Os dados da sua denominação ficam guardados nos ficheiros JSON na pasta <em>data</em> do seu próprio servidor. A única dependência do servidor central da SIGEDEC é a verificação periódica da licença (a cada 7 dias). Se o servidor central ficasse indisponível permanentemente, o sistema Standalone continuaria a funcionar normalmente até 10 dias. Passado esse prazo, o administrador pode contactar-nos para uma solução de emergência ou, em caso extremo, modificar o ficheiro de licença directamente — o que é permitido para situações de força maior devidamente documentadas.</p>

<h3>Posso criar utilizadores ilimitados na versão Standalone?</h3>
<p>Sim. A licença Standalone não impõe qualquer limite ao número de utilizadores, membros, transacções financeiras ou subdivisões. O único limite prático é a capacidade de armazenamento do seu próprio servidor — que para ficheiros JSON com dados de uma denominação típica (até 5.000 membros) nunca excederá alguns megabytes.</p>

<h3>O sistema recebe actualizações automáticas?</h3>
<p>Não — ao contrário da versão cloud, o Standalone não recebe actualizações automáticas. Quando uma nova versão do sistema for disponibilizada, os detentores de licença Standalone serão notificados por email. A actualização consiste em fazer upload dos novos ficheiros PHP para o servidor (sem afectar os dados em <em>/data</em>) e, se necessário, executar um script de migração incluído na nova versão. Correcções de segurança urgentes são comunicadas por email com instruções específicas.</p>

<h3>Posso usar o sistema offline (sem internet)?</h3>
<p>O sistema funciona mesmo sem internet para uso diário, desde que o servidor onde está instalado esteja acessível (o que acontece mesmo em hostings locais ou em redes internas). A única função que requer internet é a verificação periódica da licença. Se o seu servidor não tiver saída para a internet, contacte-nos — existe um procedimento de activação manual para instalações em redes fechadas, disponível sob pedido e verificação.</p>

<h3>Qual a diferença entre Standalone e a versão WordPress?</h3>
<p>O SIGEDEC não é um plugin ou tema WordPress — é um sistema completo e independente escrito em PHP. A analogia com o WordPress refere-se apenas ao modelo de distribuição: tal como o WordPress pode ser descarregado de wordpress.org e instalado num servidor pessoal, o SIGEDEC Standalone pode ser descarregado (após aquisição de licença) e instalado num servidor próprio. O SIGEDEC não depende do WordPress, não usa a sua base de dados, e não requer nenhum plugin ou tema adicional para funcionar.</p>
</article>

<article class="land-article">
<h2>SIGEDEC e o Google AdSense: Transparência Total</h2>
<p>O site sigedec.asaf.fyi está a candidatar-se ao programa Google AdSense para ajudar a financiar os custos de manutenção da plataforma. O Google AdSense é um programa de publicidade que exibe anúncios contextuais relevantes nos espaços publicitários do site. A receita gerada pelos anúncios ajuda a cobrir os custos de servidor, domínio e desenvolvimento contínuo do sistema.</p>

<p>É importante ser transparente: a publicidade AdSense aparecerá apenas nas páginas públicas do site de apresentação — esta página, a página de privacidade e a página de cookies. Nunca aparecerá dentro das áreas autenticadas do painel de controlo da denominação. Os membros que entram no sistema para gerir a sua congregação nunca verão anúncios.</p>

<p>Os anúncios exibidos pelo AdSense são seleccionados automaticamente pelo Google com base no conteúdo da página e no perfil de navegação do visitante. O SIGEDEC não tem controlo sobre quais anúncios específicos aparecem, mas o Google garante que os anúncios são relevantes e adequados ao contexto de um site de software de gestão eclesiástica. Qualquer anúncio inadequado pode ser reportado através dos mecanismos de feedback do próprio AdSense.</p>

<h3>Contacto e Suporte</h3>
<p>Para qualquer questão relacionada com o SIGEDEC — seja para solicitar a licença Standalone, renovar uma licença cloud, reportar um problema técnico, sugerir uma nova funcionalidade, ou simplesmente fazer uma pergunta sobre o sistema — o canal de contacto é o email audivido3@gmail.com. Não existe um sistema de tickets formal, mas todos os emails recebem resposta em menos de 48 horas. Para questões urgentes relacionadas com acesso ao sistema (por exemplo, impossibilidade de login após uma migração), o tempo de resposta habitual é de menos de 12 horas.</p>

<p>O SIGEDEC é um projecto desenvolvido em Moçambique, para igrejas africanas, por alguém que conhece de perto a realidade das denominações no continente. Não é uma empresa multinacional com um departamento de suporte — é um sistema construído com cuidado e mantido com dedicação. Cada feedback recebido é lido, considerado, e quando viável implementado nas actualizações seguintes. A voz das denominações que usam o sistema é o principal guia do seu desenvolvimento.</p>
</article>

<article class="land-article">
<h2>Gestão de Igrejas em África: O Contexto que Inspirou o SIGEDEC</h2>
<p>As igrejas e denominações cristãs em África constituem uma das maiores e mais dinâmicas comunidades religiosas do mundo. Segundo estudos do Pew Research Center, o continente africano abriga hoje mais de 700 milhões de cristãos, número que continua a crescer a um ritmo sem paralelo noutras regiões do globo. Em países como Moçambique, Angola, Congo, Nigéria, Uganda, Quénia, Tanzânia e Gana, as igrejas locais e denominações regionais têm um papel central na vida social, educativa e comunitária das suas populações.</p>

<p>No entanto, a gestão administrativa destas comunidades enfrenta desafios específicos que as soluções digitais desenvolvidas no Ocidente raramente contemplam. Em primeiro lugar, a barreira linguística: a maior parte dos sistemas de gestão de igrejas disponíveis no mercado está em Inglês, Francês ou Espanhol — línguas que muitos líderes de igrejas em contextos rurais africanos não dominam com fluidez suficiente para usar software. O SIGEDEC responde a este desafio com uma interface completamente bilingue (Português e Inglês) e com a possibilidade de adicionar outros idiomas em versões futuras.</p>

<p>Em segundo lugar, as estruturas organizacionais das denominações africanas têm características próprias que soluções genéricas não reconhecem. Uma denominação pode ter a sua sede nacional numa capital, dezenas de congregações urbanas em diferentes cidades e províncias, e centenas de postos de missão em zonas rurais. Cada nível desta hierarquia tem necessidades específicas de registo e reporte. O SIGEDEC foi desenhado com este modelo em mente: o administrador central da denominação vê tudo, cada pastor de congregação local vê apenas a sua congregação, e o tesoureiro gere as finanças com privacidade face ao restante sistema.</p>

<h3>Porque é que as Igrejas Africanas Precisam de Sistemas de Gestão Digitais</h3>
<p>Durante décadas, a gestão de membros em igrejas africanas foi feita em cadernos manuscritos, folhas de cálculo Excel, ou simplesmente na memória dos líderes mais experientes. Esta abordagem tem custos reais: membros que se deslocam de uma congregação para outra sem transferência formal, expulsões de membros que voltam a ser aceites noutras congregações sem o conhecimento do seu historial, finanças registadas de forma inconsistente ou mesmo não registadas, e incapacidade de produzir relatórios consolidados para o conselho directivo da denominação.</p>

<p>Um sistema como o SIGEDEC resolve estes problemas de forma sistemática. O registo padronizado de cada membro garante que a informação está sempre actualizada e é acessível de qualquer lugar com ligação à internet. O historial de cortes, provas e restaurações fica permanentemente registado e é consultável por qualquer administrador autorizado. As finanças são registadas com rigor e os relatórios mensais podem ser apresentados ao conselho de pastores ou à comissão financeira da denominação num formato claro e profissional.</p>

<h3>Acessibilidade Financeira como Princípio de Design</h3>
<p>O preço do SIGEDEC foi deliberadamente definido para ser acessível a denominações em contextos económicos africanos. Dez dólares por ano — o equivalente ao preço de dois cafés numa capital europeia — representa para muitas igrejas africanas um investimento significativo que precisa de ser justificado. É por isso que o sistema oferece seis meses de uso completamente gratuito antes de qualquer decisão de pagamento: a denominação tem tempo para experimentar, formar os utilizadores, migrar os dados de sistemas anteriores e avaliar com calma se o investimento faz sentido para a sua realidade.</p>

<p>Para igrejas em situação económica particularmente difícil — missões em zonas rurais remotas, denominações recentemente estabelecidas, ou comunidades afectadas por crises humanitárias — o administrador da plataforma analisa cada caso individualmente e procura soluções adaptadas. O objectivo nunca foi construir um negócio altamente lucrativo, mas sim criar uma ferramenta sustentável que sirva a igreja africana durante muitos anos. A sustentabilidade é garantida pelas licenças pagas e, futuramente, pela receita de publicidade contextual no site público.</p>

<h3>Contribuição para a Transparência e Boa Governação Eclesiástica</h3>
<p>Um dos efeitos colaterais positivos da adopção de um sistema de gestão digital como o SIGEDEC é a melhoria da transparência e da boa governação dentro das denominações. Quando as finanças estão registadas num sistema partilhado e os relatórios são gerados automaticamente, torna-se mais difícil para qualquer indivíduo — seja um tesoureiro, um pastor local ou um administrador regional — manipular ou ocultar informação. A rastreabilidade inerente ao sistema digital promove naturalmente a responsabilização e a confiança entre líderes e membros.</p>

<p>Da mesma forma, o registo cuidadoso do historial de membros — incluindo cortes, provas, restaurações e transferências — contribui para uma gestão pastoral mais consciente e fundamentada. Em vez de decisões baseadas em memórias selectivas ou em informação de segunda mão, o conselho de liderança tem acesso a factos registados com data, testemunhas e circunstâncias documentadas. Esta cultura de registo rigoroso, quando adoptada de forma consistente, eleva o nível de profissionalismo e de credibilidade da denominação perante os seus membros e perante a sociedade em geral.</p>

<p>O SIGEDEC não é apenas um software. É uma resposta concreta a um problema real, desenvolvida por alguém que conhece por dentro a realidade das igrejas africanas e que acredita que a tecnologia pode servir eficazmente a missão e a organização das comunidades de fé no continente africano. Cada funcionalidade foi pensada a partir de situações reais: a dificuldade de saber quantos membros activos tem uma denominação espalhada por várias províncias, a necessidade de verificar o historial de um membro antes de o receber em transferência, a importância de ter relatórios financeiros claros para apresentar ao conselho directivo. O SIGEDEC existe para tornar estas tarefas simples, rápidas e fiáveis — para que a liderança eclesiástica possa concentrar a sua atenção naquilo que mais importa: o acompanhamento pastoral, o crescimento espiritual e o serviço às comunidades.</p>
</article>

<article class="land-article">
<h2>Como Registar a Sua Denominação no SIGEDEC</h2>
<p>O registo de uma nova denominação no SIGEDEC é um processo simples que demora menos de três minutos. Aceda a sigedec.asaf.fyi e clique no botão "Registar Denominação" no cabeçalho da página. Será apresentado um formulário com os seguintes campos: nome completo da denominação, país, província, distrito, nome do administrador principal, endereço de email do administrador, e senha de acesso.</p>

<p>O nome da denominação deve ser o nome oficial completo — por exemplo "Assembleia de Deus em Moçambique — Sector Sul" ou "Igreja Universal do Reino de Deus — Delegação de Nampula". Este nome aparecerá em todos os relatórios, na lista pública de denominações registadas, e no cabeçalho do painel de controlo. Pode ser alterado posteriormente nas configurações.</p>

<p>O endereço de email serve como identificador de conta e como meio de comunicação para receber chaves de licença, actualizações importantes e respostas a questões de suporte. Recomenda-se usar um email institucional da denominação em vez de um email pessoal, para garantir continuidade mesmo em caso de mudança de liderança administrativa.</p>

<p>A senha deve ter pelo menos seis caracteres e incluir pelo menos uma letra maiúscula, uma letra minúscula, um número e um símbolo especial. Uma senha forte protege o acesso a informação sensível sobre os membros da denominação — o mesmo nível de cuidado que se teria com um arquivo físico contendo os processos individuais de cada membro. Após submeter o formulário, a denominação fica registada de imediato e o período de teste gratuito de seis meses começa automaticamente. Não é enviado email de confirmação — o acesso ao sistema está disponível no momento do registo, usando as credenciais definidas no formulário.</p>

<h3>Sugestões e Desenvolvimento Participativo</h3>
<p>O SIGEDEC é um sistema em desenvolvimento activo. Cada versão inclui melhorias sugeridas por denominações que usam o sistema no dia-a-dia. Se a sua denominação identificar uma funcionalidade em falta, um comportamento que poderia ser melhorado, ou uma secção que não faz sentido na sua realidade específica, a sugestão é bem-vinda e será analisada com atenção. O canal para enviar sugestões é o email audivido3@gmail.com, com o assunto "Sugestão SIGEDEC". Descreva o que gostaria de ver diferente, porquê, e como utilizaria essa funcionalidade no dia-a-dia da sua denominação. As sugestões mais solicitadas são priorizadas nas actualizações seguintes e os autores das sugestões implementadas são informados por email quando a funcionalidade fica disponível.</p>
</article>

</div>
</section>

<div class="land-suggestion-strip">
    <div class="land-container">
        <div class="land-suggestion-inner">
            <div class="land-sugg-icon">💡</div>
            <div class="land-sugg-text">
                <strong><?= $lang==='en' ? 'Help us improve SIGEDEC!' : 'Ajude-nos a melhorar o SIGEDEC!' ?></strong>
                <span><?= $lang==='en'
                    ? 'Does your denomination think the system could be better? Suggest a new feature, request the removal of something or report any problem.'
                    : 'A sua denominação acha que o sistema devia melhorar, adicionar uma certa funcionalidade ou remover algo? Envie a sua sugestão — cada ideia conta.' ?></span>
            </div>
            <a href="mailto:<?= CONTACT_EMAIL ?>?subject=<?= urlencode($lang==='en' ? 'SIGEDEC — Suggestion' : 'SIGEDEC — Sugestão') ?>" class="land-sugg-btn">
                <?= $lang==='en' ? '✉️ Send suggestion' : '✉️ Enviar sugestão' ?>
            </a>
        </div>
    </div>
</div>

<footer class="land-footer">
    <div class="land-container">
        <div class="land-footer-grid">
            <div class="land-footer-brand">
                <div class="land-logo"><span class="land-cross">✝</span><span class="land-logo-text">SIGEDEC</span></div>
                <p><?= $lang==='en' ? 'Church Management System for Africa' : 'Sistema de Gestão Eclesiástica para África' ?></p>
                <p style="font-size:0.75rem;color:rgba(255,255,255,0.25);margin-top:0.5rem"><?= SITE_URL ?></p>
            </div>
            <div class="land-footer-links">
                <h4><?= $lang==='en' ? 'System' : 'Sistema' ?></h4>
                <a href="register.php"><?= __('register') ?></a>
                <a href="public/"><?= $lang==='en' ? 'Public Directory' : 'Directório Público' ?></a>
                <a href="#funcionalidades"><?= $lang==='en' ? 'Features' : 'Funcionalidades' ?></a>
                <a href="#tutorial">Tutorial</a>
            </div>
            <div class="land-footer-links">
                <h4><?= $lang==='en' ? 'Legal' : 'Legal' ?></h4>
                <a href="privacy.php"><?= __('privacy_policy') ?></a>
                <a href="cookies.php"><?= __('cookie_policy') ?></a>
                <a href="#precos"><?= $lang==='en' ? 'Pricing' : 'Preços' ?></a>
                <a href="#standalone">Standalone</a>
            </div>
            <div class="land-footer-links">
                <h4><?= $lang==='en' ? 'Contact' : 'Contacto' ?></h4>
                <a href="mailto:<?= CONTACT_EMAIL ?>"><?= CONTACT_EMAIL ?></a>
                <a href="?lang=<?= __('lang_code') ?>">🌐 <?= __('lang_switch') ?></a>
            </div>
        </div>
        <div class="land-footer-bottom">
            <span>© <?= $year ?> SIGEDEC — <?= $lang==='en' ? 'All rights reserved' : 'Todos os direitos reservados' ?></span>
            <span><?= $lang==='en' ? 'Made with ✝ for Africa' : 'Feito com ✝ para África' ?></span>
            <span><?= $day ?> <?= $monthName ?> <?= $year ?></span>
        </div>
    </div>
</footer>

<!-- ═══ COOKIE NOTICE ═════════════════════════════════════════════ -->
<div class="cookie-notice" id="cookieNotice" style="display:none">
    <div class="cookie-inner">
        <p><?= __('cookie_msg') ?><a href="cookies.php"><?= __('cookie_policy') ?></a> <?= $lang==='en' ? 'and our' : 'e a nossa' ?> <a href="privacy.php"><?= __('privacy_policy') ?></a>.</p>
        <div class="cookie-btns">
            <button onclick="acceptCookies()" class="btn btn-primary btn-sm"><?= __('cookie_accept') ?></button>
            <button onclick="declineCookies()" class="btn btn-outline btn-sm"><?= __('cookie_decline') ?></button>
        </div>
    </div>
</div>

<script>
// Mobile menu
document.getElementById('landMenuBtn').addEventListener('click', () => {
    document.getElementById('landMobileNav').classList.toggle('open');
});
// Password toggle
function togglePass() {
    const i = document.getElementById('passInput');
    i.type = i.type === 'password' ? 'text' : 'password';
}
// Cookies
function acceptCookies() {
    localStorage.setItem('sigedec_cookies','accepted');
    document.getElementById('cookieNotice').style.display='none';
}
function declineCookies() {
    localStorage.setItem('sigedec_cookies','declined');
    document.getElementById('cookieNotice').style.display='none';
}
window.addEventListener('load', () => {
    if (!localStorage.getItem('sigedec_cookies')) {
        document.getElementById('cookieNotice').style.display='flex';
    }
});
// Scroll active nav
const sections = document.querySelectorAll('section[id]');
const navLinks = document.querySelectorAll('.land-nav a');
window.addEventListener('scroll', () => {
    let cur = '';
    sections.forEach(s => { if (window.scrollY >= s.offsetTop - 100) cur = s.id; });
    navLinks.forEach(a => { a.classList.toggle('active', a.getAttribute('href') === '#' + cur); });
});
// Auto-dismiss alerts
setTimeout(() => {
    document.querySelectorAll('.alert').forEach(a => {
        a.style.transition='opacity 0.5s'; a.style.opacity='0';
        setTimeout(()=>a.remove(),500);
    });
}, 5000);

// FAQ accordion
function toggleFaq(btn) {
    const item = btn.closest('.faq-item');
    const ans  = item.querySelector('.faq-a');
    const isOpen = btn.classList.contains('open');
    // Fechar todos do mesmo bloco
    btn.closest('.faq-block').querySelectorAll('.faq-q').forEach(b => {
        b.classList.remove('open');
        b.closest('.faq-item').querySelector('.faq-a').classList.remove('open');
    });
    if (!isOpen) {
        btn.classList.add('open');
        ans.classList.add('open');
    }
}
</script>
</body>
</html>
